import PDFDocument from 'pdfkit';

const titleFontSize = 24;
const subTitleFontSize = 14;
const leadingFontSize = 13;
const contentFontSize = 11;
const tableHeaderFontSize = 12;
const footerFontSize = 10;

const paperSizeHeight = 841.89;
const paperSizeWidth = 595.28;
const paperMargin = 40;
const lineSpacing = 18;
const bottomGap = 25;

export async function createMasterInvoice(invoice) {
  let doc = new PDFDocument({
    size: [paperSizeWidth, paperSizeHeight],
    margin: paperMargin,
  });

  generateHeader(doc, invoice, 60);
  await generateOrderInformation(doc, invoice, 210);
  generateInvoiceTable(doc, invoice, 330);
  generateFooter(doc);
  return doc;
}

function generateHeader(doc, invoice, top) {
  const leftX = paperMargin;
  const rightX = paperSizeWidth - paperMargin - 150;

  doc
    .fillColor('#2c3e50')
    .fontSize(titleFontSize)
    .font('Helvetica-Bold')
    .text('INVOICE', { align: 'center' })
    .moveDown(0.5)
    .fontSize(subTitleFontSize)
    .font('Helvetica')
    .fillColor('#000');

  // Left Column - Company Info
  doc
    .fontSize(leadingFontSize)
    .font('Helvetica-Bold')
    .fillColor('blue') 
    .text(invoice.invoiceNumber, leftX, top + lineSpacing * 2)
    .fillColor('black') 
    .text('Webapps Software Solutions', leftX, top + lineSpacing * 3)
    .font('Helvetica')
    .text('128, Jodhpur Gardens, Lake Gardens,', leftX, top + lineSpacing * 4)
    .text('Kolkata, West Bengal 700045', leftX, top + lineSpacing * 5)
    .text('GST No.: 19CRLPP7937D1Z7', leftX, top + lineSpacing * 6);

  // Right Column - Client Info
  doc
    .fontSize(contentFontSize)
    .font('Helvetica-Bold')
    .fillColor('red') 
    .text('Bill To:', rightX, top + lineSpacing * 2)
    .fillColor('black') 
    .font('Helvetica-Bold')
    .text(
      invoice.businessName || invoice?.business?.personName,
      rightX,
      top + lineSpacing * 3,
    )
    .font('Helvetica')
    .text(
      `GST No.: ${invoice?.business?.gstNo || 'N/A'}`,
      rightX,
      top + 30 + lineSpacing * 4,
    );
}

async function generateOrderInformation(doc, invoice, top) {
  const rightX = paperSizeWidth - paperMargin - 350;

  doc
    .fontSize(contentFontSize)
    .font('Helvetica-Bold')
    .text(`Invoice Date: ${invoice.paymentDate}`, rightX, top, {
      align: 'right',
    })
    .text(
      `Sales Agent: Webapps Software Solutions`,
      rightX,
      top + lineSpacing,
      { align: 'right' },
    )
    .text(`Subject: ${invoice.subject}`, rightX, top + lineSpacing * 2, {
      align: 'right',
    });

  // generateHr(doc, top + lineSpacing * 4 + bottomGap);
}

function generateInvoiceTable(doc, invoice, y) {
  const labelX = 350;
  const valueX = 500;

  doc.font('Helvetica-Bold').fontSize(tableHeaderFontSize);
  generateTableRow(doc, y, 'Item', 'Qty', 'Tax', 'Amount', true);
  generateHr(doc, y + lineSpacing);

  doc.font('Helvetica').fontSize(contentFontSize);
  let position = y + lineSpacing * 2;
  generateTableRow(
    doc,
    position,
    invoice.subject,
    '1',
    '',
    invoice.subTotal.toFixed(2),
  );

  let taxPosition = position + lineSpacing;
  invoice.tax.forEach((tax, index) => {
    generateTableRow(
      doc,
      taxPosition + index * lineSpacing,
      '',
      '',
      `${tax.taxName}: ${tax.rate}%`,
      ((invoice.subTotal * parseFloat(tax.rate)) / 100).toFixed(2),
    );
  });

  taxPosition += invoice.tax.length * lineSpacing + 20;

  doc.font('Helvetica-Bold');
  generateAlignedText(doc, taxPosition, 'Sub Total:', invoice.subTotal, labelX, valueX);
  generateAlignedText(doc, taxPosition + lineSpacing, 'Total:', invoice.total, labelX, valueX);
  generateAlignedText(doc, taxPosition + lineSpacing * 2, 'Total Paid:', invoice.paid, labelX, valueX);
  generateAlignedText(doc, taxPosition + lineSpacing * 3, 'Amount Due:', invoice.amountDue, labelX, valueX);
}

function generateAlignedText(doc, y, label, amount, labelX, valueX) {
  doc
    .fontSize(contentFontSize)
    .text(label, labelX, y, { width: valueX - labelX - 10, align: 'right' })
    .text(formatCurrency(amount), valueX, y, { align: 'right' });
}

function generateTableRow(doc, y, item, qty, tax, amount, isHeader = false) {
  const fontStyle = isHeader ? 'Helvetica-Bold' : 'Helvetica';
  const fontSize = isHeader ? tableHeaderFontSize : contentFontSize;

  doc
    .font(fontStyle)
    .fontSize(fontSize)
    .text(item, paperMargin, y, { width: 180 })
    .text(qty, paperMargin + 200, y, { width: 80, align: 'center' })
    .text(tax, paperMargin + 300, y, { width: 100, align: 'center' })
    .text(amount, paperMargin + 430, y, { width: 80, align: 'right' });
}

function generateTableTotal(doc, y, title, total, emphasize = false) {
  doc
    .fontSize(contentFontSize)
    .font(emphasize ? 'Helvetica-Bold' : 'Helvetica')
    .text(title, 300, y, { width: 200, align: 'right' })
    .text(total, 510, y, { width: 60, align: 'right' });
}

function generateFooter(doc) {
  const footerText =
    'This is a computer-generated invoice and requires no signature or stamp.';
  const footerY = doc.page.height - doc.page.margins.bottom - 30;

  doc
    .fontSize(footerFontSize)
    .fillColor('#555')
    .text(footerText, doc.page.margins.left, footerY, {
      width: doc.page.width - doc.page.margins.left - doc.page.margins.right,
      align: 'center',
    });
}

function formatCurrency(amount) {
  return `${parseFloat(amount).toFixed(2)}`;
}

function generateHr(doc, y) {
  doc
    .strokeColor('#cccccc')
    .lineWidth(1)
    .moveTo(paperMargin, y)
    .lineTo(paperSizeWidth - paperMargin, y)
    .stroke();
}
