import { MailerService } from '@nestjs-modules/mailer';
import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { BusinessEmail } from 'src/business-email/entities/business-email.entity';
import { Repository } from 'typeorm';
import * as nodemailer from 'nodemailer';
import { DefaultStatus } from 'src/enum';

@Injectable()
export class NodeMailerService {
  constructor(
    // private readonly mailService: MailerService,
    @InjectRepository(BusinessEmail)
    private readonly busiEmailRepo: Repository<BusinessEmail>,
  ) {}

  async userRegisterMail(email, businessAccId) {
    if (!email || !businessAccId) {
      console.warn('Missing email or PDF buffer for invoice sending.');
      return false;
    }
    try {
      const smtpConfig = await this.getSmtpConfigOfBusiness(businessAccId);
      const transporter = nodemailer.createTransport({
        host: smtpConfig.host,
        port: smtpConfig.port,
        secure: smtpConfig.secure,
        auth: {
          user: smtpConfig.user,
          pass: smtpConfig.pass,
        },
        tls: {
          rejectUnauthorized: true,
        },
      });
      await transporter.sendMail({
        from: smtpConfig.from,
        to: email,
        subject: 'Successful Registration',
        html: `<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Welcome to Privilege</title>
  <style>
    body {
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
      background-color: #ffffff;
      color: #333333;
      margin: 0;
      padding: 0;
    }
    .container {
      max-width: 600px;
      margin: 30px auto;
      padding: 20px;
      border: 1px solid #e0e0e0;
      border-radius: 10px;
      background-color: #ffffff;
    }
    .logo {
      text-align: center;
      margin-bottom: 20px;
    }
    .logo img {
      max-height: 60px;
    }
    h1 {
      color: #2c3e50;
      font-size: 24px;
    }
    p {
      font-size: 16px;
      line-height: 1.6;
    }
    .footer {
      font-size: 12px;
      color: #888;
      text-align: center;
      margin-top: 30px;
    }
    .footer a {
      color: #888;
      text-decoration: none;
    }
  </style>
</head>
<body>
  <div class="container">
    <!-- <div class="logo">
      <img src="Logo.png" alt="Privilege Logo">
    </div> -->
    <h1>Welcome to Privilege!</h1>
    <p>Dear User,</p>
    <p>Thank you for registering with <strong>Privilege</strong>. We're excited to have you onboard. Your journey toward exclusive member benefits starts now.</p>
    <p>Our team will review your account shortly. You’ll be notified once it’s approved or if any further action is needed.</p>
    <p>Stay tuned and feel free to reach out if you have any questions.</p>
    <p>Warm regards,<br><strong>The Privilege Team</strong></p>
    <div class="footer">
      <p>Privilege Inc., 1234 Any Place, Kolkata, India</p>
      <p>Email: support@privilege.com | Phone: +123-456-7890</p>
      <p><a href="#">Unsubscribe</a></p>
    </div>
  </div>
</body>
</html>`,
      });
      return true;
    } catch (error) {
      console.error('Mail send failed:', error);
      return false;
    }
  }

  async userAccountReject(email, businessAccId) {
    if (!email || !businessAccId) {
      return false;
    }
    try {
      const smtpConfig = await this.getSmtpConfigOfBusiness(businessAccId);
      const transporter = nodemailer.createTransport({
        host: smtpConfig.host,
        port: smtpConfig.port,
        secure: smtpConfig.secure,
        auth: {
          user: smtpConfig.user,
          pass: smtpConfig.pass,
        },
        tls: {
          rejectUnauthorized: true,
        },
      });
      await transporter.sendMail({
        from: smtpConfig.from,
        to: email,
        subject: 'Account rejected!',
        html: `<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Account Rejection Notification</title>
  <style>
    body {
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
      background-color: #ffffff;
      color: #333333;
      margin: 0;
      padding: 0;
    }
    .container {
      max-width: 600px;
      margin: 30px auto;
      padding: 20px;
      border: 1px solid #e0e0e0;
      border-radius: 10px;
      background-color: #ffffff;
    }
    .logo {
      text-align: center;
      margin-bottom: 20px;
    }
    .logo img {
      max-height: 60px;
    }
    h1 {
      color: #c0392b;
      font-size: 22px;
    }
    p {
      font-size: 16px;
      line-height: 1.6;
    }
    .footer {
      font-size: 12px;
      color: #888;
      text-align: center;
      margin-top: 30px;
    }
    .footer a {
      color: #888;
      text-decoration: none;
    }
  </style>
</head>
<body>
  <div class="container">
    <!-- <div class="logo">
      <img src="Logo.png" alt="Privilege Logo">
    </div> -->
    <h1>Account Rejected</h1>
    <p>Dear User,</p>
    <p>We regret to inform you that your account registration with <strong>Privilege</strong> has been rejected after review.</p>
    <p>This may be due to missing or invalid information. You may contact our support team for clarification or try registering again with valid details.</p>
    <p>We appreciate your interest in Privilege and hope to assist you in the future.</p>
    <p>Sincerely,<br><strong>The Privilege Team</strong></p>
    <div class="footer">
      <p>Privilege Inc., 1234 Any Place, Kolkata, India</p>
      <p>Email: support@privilege.com | Phone: +123-456-7890</p>
      <p><a href="#">Unsubscribe</a></p>
    </div>
  </div>
</body>
</html>`,
      });
      return true;
    } catch (error) {
      console.log(error);
      return false;
    }
  }

  async userAccountApproval(email, businessAccId) {
    if (!email || !businessAccId) {
      return false;
    }
    try {
      const smtpConfig = await this.getSmtpConfigOfBusiness(businessAccId);
      const transporter = nodemailer.createTransport({
        host: smtpConfig.host,
        port: smtpConfig.port,
        secure: smtpConfig.secure,
        auth: {
          user: smtpConfig.user,
          pass: smtpConfig.pass,
        },
        tls: {
          rejectUnauthorized: true,
        },
      });
      await transporter.sendMail({
        from: smtpConfig.from,
        to: email,
        subject: 'Account approved',
        html: `<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Your Privilege Account is Approved</title>
  <style>
    body {
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
      background-color: #ffffff;
      color: #333333;
      margin: 0;
      padding: 0;
    }
    .container {
      max-width: 600px;
      margin: 30px auto;
      padding: 20px;
      border: 1px solid #e0e0e0;
      border-radius: 10px;
      background-color: #ffffff;
    }
    .logo {
      text-align: center;
      margin-bottom: 20px;
    }
    .logo img {
      max-height: 60px;
    }
    h1 {
      color: #2c3e50;
      font-size: 24px;
    }
    p {
      font-size: 16px;
      line-height: 1.6;
    }
    .footer {
      font-size: 12px;
      color: #888;
      text-align: center;
      margin-top: 30px;
    }
    .footer a {
      color: #888;
      text-decoration: none;
    }
  </style>
</head>
<body>
  <div class="container">
    <!-- <div class="logo">
      <img src="Logo.png" alt="Privilege Logo">
    </div> -->
    <h1>Your Account is Approved!</h1>
    <p>Dear User,</p>
    <p>We’re happy to inform you that your account with <strong>Privilege</strong> has been successfully approved.</p>
    <p>You can now access your dashboard and explore your exclusive membership options and benefits.</p>
    <p>Thank you for being a part of Privilege.</p>
    <p>Best regards,<br><strong>The Privilege Team</strong></p>
    <div class="footer">
      <p>Privilege Inc., 1234 Any Place, Kolkata, India</p>
      <p>Email: support@privilege.com | Phone: +123-456-7890</p>
      <p><a href="#">Unsubscribe</a></p>
    </div>
  </div>
</body>
</html>`,
      });
      return true;
    } catch (error) {
      console.log(error);
      return false;
    }
  }

  //here send app link in mail body
  async paymentLinkAfterApprovalUser(email, businessAccId) {
    if (!email || !businessAccId) {
      return false;
    }
    try {
      const smtpConfig = await this.getSmtpConfigOfBusiness(businessAccId);
      const transporter = nodemailer.createTransport({
        host: smtpConfig.host,
        port: smtpConfig.port,
        secure: smtpConfig.secure,
        auth: {
          user: smtpConfig.user,
          pass: smtpConfig.pass,
        },
        tls: {
          rejectUnauthorized: true,
        },
      });
      await transporter.sendMail({
        from: smtpConfig.from,
        to: email,
        subject: 'Payment link for membership card buy',
        html: `<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Complete Your Privilege Membership</title>
  <style>
    body {
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
      background-color: #ffffff;
      color: #333333;
      margin: 0;
      padding: 0;
    }
    .container {
      max-width: 600px;
      margin: 30px auto;
      padding: 20px;
      border: 1px solid #e0e0e0;
      border-radius: 10px;
      background-color: #ffffff;
    }
    .logo {
      text-align: center;
      margin-bottom: 20px;
    }
    .logo img {
      max-height: 60px;
    }
    h1 {
      color: #2c3e50;
      font-size: 24px;
    }
    p {
      font-size: 16px;
      line-height: 1.6;
    }
    .button {
      display: inline-block;
      background-color: #1a73e8;
      color: white;
      padding: 12px 24px;
      border-radius: 5px;
      text-decoration: none;
      font-weight: bold;
      margin-top: 20px;
    }
    .footer {
      font-size: 12px;
      color: #888;
      text-align: center;
      margin-top: 30px;
    }
    .footer a {
      color: #888;
      text-decoration: none;
    }
  </style>
</head>
<body>
  <div class="container">
    <!-- <div class="logo">
      <img src="Logo.png" alt="Privilege Logo">
    </div> -->
    <h1>Complete Your Membership</h1>
    <p>Dear User,</p>
    <p>You're just one step away from unlocking your exclusive Privilege benefits. Please complete your payment to activate your membership card.</p>
    <p>Click the button below to proceed with payment:</p>
    <p><a href="https://yourpaymentlink.com" class="button">Pay Now</a></p>
    <p>If you have any questions or face issues during payment, feel free to contact our support team.</p>
    <p>Best regards,<br><strong>The Privilege Team</strong></p>
    <div class="footer">
      <p>Privilege Inc., 1234 Any Place, Kolkata, India</p>
      <p>Email: support@privilege.com | Phone: +123-456-7890</p>
      <p><a href="#">Unsubscribe</a></p>
    </div>
  </div>
</body>
</html>
`,
      });
      return true;
    } catch (error) {
      console.log(error);
      return false;
    }
  }

  async userPaymentSuccess(email, businessAccId) {
    if (!email || !businessAccId) {
      return false;
    }
    try {
      const smtpConfig = await this.getSmtpConfigOfBusiness(businessAccId);
      const transporter = nodemailer.createTransport({
        host: smtpConfig.host,
        port: smtpConfig.port,
        secure: smtpConfig.secure,
        auth: {
          user: smtpConfig.user,
          pass: smtpConfig.pass,
        },
        tls: {
          rejectUnauthorized: true,
        },
      });
      await transporter.sendMail({
        from: smtpConfig.from,
        to: email,
        subject: 'Payment successful',
        html: `<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Payment Successful - Privilege Membership</title>
  <style>
    body {
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
      background-color: #ffffff;
      color: #333333;
      margin: 0;
      padding: 0;
    }
    .container {
      max-width: 600px;
      margin: 30px auto;
      padding: 20px;
      border: 1px solid #e0e0e0;
      border-radius: 10px;
      background-color: #ffffff;
    }
    .logo {
      text-align: center;
      margin-bottom: 20px;
    }
    .logo img {
      max-height: 60px;
    }
    h1 {
      color: #2c3e50;
      font-size: 24px;
    }
    p {
      font-size: 16px;
      line-height: 1.6;
    }
    .footer {
      font-size: 12px;
      color: #888;
      text-align: center;
      margin-top: 30px;
    }
    .footer a {
      color: #888;
      text-decoration: none;
    }
  </style>
</head>
<body>
  <div class="container">
    <!-- <div class="logo">
      <img src="Logo.png" alt="Privilege Logo">
    </div> -->
    <h1>Payment Successful!</h1>
    <p>Dear User,</p>
    <p>We have successfully received your payment for the Privilege membership card.</p>
    <p>Your membership will be activated shortly and the card details will be sent to you in a separate email.</p>
    <p>Thank you for choosing Privilege. We're excited to have you with us!</p>
    <p>Sincerely,<br><strong>The Privilege Team</strong></p>
    <div class="footer">
      <p>Privilege Inc., 1234 Corporate Ave, City, Country</p>
      <p>Email: support@privilege.com | Phone: +123-456-7890</p>
      <p><a href="#">Unsubscribe</a></p>
    </div>
  </div>
</body>
</html>
`,
      });
      return true;
    } catch (error) {
      console.log(error);
      return false;
    }
  }

  async userPaymentInvoice(email, memberId, pdfBuffer, businessAccId) {
    if (!email || !pdfBuffer || !businessAccId) {
      console.warn('Missing email or PDF buffer for invoice sending.');
      return false;
    }
    try {
      const smtpConfig = await this.getSmtpConfigOfBusiness(businessAccId);
      const transporter = nodemailer.createTransport({
        host: smtpConfig.host,
        port: smtpConfig.port,
        secure: smtpConfig.secure,
        auth: {
          user: smtpConfig.user,
          pass: smtpConfig.pass,
        },
        tls: {
          rejectUnauthorized: true,
        },
      });
      await transporter.sendMail({
        from: smtpConfig.from,
        to: email,
        subject: 'Invoice of Membership card Purchase',
        html: ` <p>Dear Member,</p>
        <p>Thank you for your membership purchase. Please find your invoice attached.</p>
        <p>If you have any questions, feel free to contact us.</p>
        <br/>
        <p>Regards,<br/>Privilege</p>`,
        attachments: [
          {
            filename: `invoice_${memberId}.pdf`,
            content: pdfBuffer,
          },
        ],
      });
      return true;
    } catch (error) {
      console.error('Mail send failed:', error);
      return false;
    }
  }

  async userPaymentFailed(email, businessAccId) {
    if (!email || !businessAccId) {
      return;
    }
    try {
      const smtpConfig = await this.getSmtpConfigOfBusiness(businessAccId);
      const transporter = nodemailer.createTransport({
        host: smtpConfig.host,
        port: smtpConfig.port,
        secure: smtpConfig.secure,
        auth: {
          user: smtpConfig.user,
          pass: smtpConfig.pass,
        },
        tls: {
          rejectUnauthorized: true,
        },
      });
      await transporter.sendMail({
        from: smtpConfig.from,
        to: email,
        subject: 'Payment failed',
        html: `<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Payment Failed</title>
  <style>
    body {
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
      background-color: #ffffff;
      margin: 0;
      padding: 0;
      color: #333;
    }
    .container {
      width: 100%;
      max-width: 600px;
      margin: auto;
      padding: 20px;
    }
    .header {
      text-align: center;
      padding: 10px;
    }
    .header img {
      max-height: 60px;
    }
    .content {
      background-color: #ffffff;
      border: 1px solid #e0e0e0;
      padding: 30px;
      border-radius: 8px;
    }
    .button {
      display: inline-block;
      margin-top: 20px;
      padding: 12px 24px;
      background-color: #dc3545;
      color: #ffffff;
      text-decoration: none;
      border-radius: 6px;
    }
    .footer {
      font-size: 12px;
      color: #888;
      text-align: center;
      margin-top: 20px;
      padding: 10px;
    }
    a {
      color: #0056b3;
      text-decoration: none;
    }
  </style>
</head>
<body>
  <div class="container">
    <!-- <div class="header">
      <img src="Logo.png" alt="Privilege Logo">
    </div> -->
    <div class="content">
      <h2>Payment Failed</h2>
      <p>Dear User,</p>
      <p>Unfortunately, your payment for the Privilege membership card was unsuccessful.</p>
      <p>Please retry your payment by clicking the button below:</p>
      <p><a href="https://yourpaymentretrylink.com" class="button">Retry Payment</a></p>
      <p>If you continue to face issues, contact our support team for assistance.</p>
      <p>Sincerely,<br>The Privilege Team</p>
    </div>
    <div class="footer">
      <p>Privilege Inc., 1234 Any Place, Kolkata, India</p>
      <p>Contact us: support@privilege.com | +1 (234) 567-890</p>
      <p><a href="#">Unsubscribe</a></p>
    </div>
  </div>
</body>
</html>
`,
      });
      return true;
    } catch (error) {
      console.error('Mail send failed:', error);
      return false;
    }
  }

  async sendMembershipCardDetail(
    email,
    cardNo,
    cardName,
    activedate,
    expDate,
    businessAccId,
  ) {
    if (!email || !businessAccId) {
      return false;
    }
    try {
      const smtpConfig = await this.getSmtpConfigOfBusiness(businessAccId);
      const transporter = nodemailer.createTransport({
        host: smtpConfig.host,
        port: smtpConfig.port,
        secure: smtpConfig.secure,
        auth: {
          user: smtpConfig.user,
          pass: smtpConfig.pass,
        },
        tls: {
          rejectUnauthorized: true,
        },
      });
      await transporter.sendMail({
        from: smtpConfig.from,
        to: email,
        subject: 'Details of Membership Card',
        html: `<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Your Privilege Membership Details</title>
  <style>
    body {
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
      background-color: #ffffff;
      margin: 0;
      padding: 0;
      color: #333;
    }
    .container {
      width: 100%;
      max-width: 600px;
      margin: auto;
      padding: 20px;
    }
    .header {
      text-align: center;
      padding: 10px;
    }
    .header img {
      max-height: 60px;
    }
    .content {
      background-color: #ffffff;
      border: 1px solid #e0e0e0;
      padding: 30px;
      border-radius: 8px;
    }
    .card-info {
      margin-top: 20px;
      background-color: #f8f9fa;
      padding: 15px;
      border-radius: 6px;
      border: 1px solid #ccc;
    }
    .card-info p {
      margin: 6px 0;
    }
    .footer {
      font-size: 12px;
      color: #888;
      text-align: center;
      margin-top: 20px;
      padding: 10px;
    }
    a {
      color: #0056b3;
      text-decoration: none;
    }
  </style>
</head>
<body>
  <div class="container">
    <!-- <div class="header">
      <img src="Logo.png" alt="Privilege Logo">
    </div> -->
    <div class="content">
      <h2>Your Membership Card Details</h2>
      <p>Dear User,</p>
      <p>Thank you for purchasing your Privilege membership. Below are the details of your card:</p>
      <div class="card-info">
        <p><strong>Membership Card:</strong> ${cardName}</p>
        <p><strong>Card Number:</strong> ${cardNo}</p>
        <p><strong>Activation Date:</strong> ${activedate}</p>
        <p><strong>Expiry Date:</strong> ${expDate}</p>
      </div>
      <p>Please keep this information for your records. If you have any questions, feel free to contact us.</p>
      <p>Warm regards,<br>The Privilege Team</p>
    </div>
    <div class="footer">
      <p>Privilege Inc., 1234 Any Place, Kolkata, India</p>
      <p>Contact us: support@privilege.com | +1 (234) 567-890</p>
      <p><a href="#">Unsubscribe</a></p>
    </div>
  </div>
</body>
</html>`,
      });
      return true;
    } catch (error) {
      console.error('Mail send failed:', error);
      return false;
    }
  }

  async sendMembershipExpiringMail(
    email,
    userName,
    cardNo,
    cardName,
    expDate,
    businessAccId,
  ) {    
    if (!email || !businessAccId) {
      return false;
    }
    try {
      const smtpConfig = await this.getSmtpConfigOfBusiness(businessAccId);
      const transporter = nodemailer.createTransport({
        host: smtpConfig.host,
        port: smtpConfig.port,
        secure: smtpConfig.secure,
        auth: {
          user: smtpConfig.user,
          pass: smtpConfig.pass,
        },
        tls: {
          rejectUnauthorized: true,
        },
      });
      await transporter.sendMail({
        from: smtpConfig.from,
        to: email,
        subject: 'Privilege Membership is Expiring Soon',
        html: `<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Your Privilege Membership is Expiring Soon</title>
  <style>
    body {
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
      background-color: #ffffff;
      margin: 0;
      padding: 0;
      color: #333;
    }
    .container {
      width: 100%;
      max-width: 600px;
      margin: auto;
      padding: 20px;
    }
    .header {
      text-align: center;
      padding: 10px;
    }
    .header img {
      max-height: 60px;
    }
    .content {
      background-color: #ffffff;
      border: 1px solid #e0e0e0;
      padding: 30px;
      border-radius: 8px;
    }
    .card-info {
      margin-top: 20px;
      background-color: #f8f9fa;
      padding: 15px;
      border-radius: 6px;
      border: 1px solid #ccc;
    }
    .card-info p {
      margin: 6px 0;
    }
    .footer {
      font-size: 12px;
      color: #888;
      text-align: center;
      margin-top: 20px;
      padding: 10px;
    }
    a {
      color: #0056b3;
      text-decoration: none;
    }
    .btn-renew {
      display: inline-block;
      margin-top: 15px;
      padding: 10px 16px;
      background-color: #0056b3;
      color: #ffffff !important;
      border-radius: 4px;
      text-decoration: none;
      font-size: 14px;
    }
  </style>
</head>
<body>
  <div class="container">
    <div class="content">
      <h2>Your Membership Card is Expiring Soon</h2>
      <p>Dear ${userName},</p>
      <p>This is a friendly reminder that your Privilege membership card will expire soon. We encourage you to renew your membership card to continue enjoying exclusive benefits.</p>
      
      <div class="card-info">
        <p><strong>Membership Card Name:</strong> ${cardName}</p>
        <p><strong>Card Number:</strong> ${cardNo}</p>
        <p><strong>Expiry Date:</strong> ${expDate}</p>
      </div>

      <p>Renewing your membership ensures uninterrupted access to all features and rewards.</p>

      <p style="margin-top: 20px;">If you have already renewed, please ignore this message.</p>
      <p>Warm regards,<br>The Privilege Team</p>
    </div>

    <div class="footer">
      <p>Privilege Inc., 1234 Any Place, Kolkata, India</p>
    </div>
  </div>
</body>
</html>
`,
      });
      return true;
    } catch (error) {
      console.error('Mail send failed:', error);
      return false;
    }
  }

  async sendOtpInMail(email, otp, businessAccId: string) {
    if (!email || !businessAccId) {
      return false;
    }
    try {
      const smtpConfig = await this.getSmtpConfigOfBusiness(businessAccId);
      const transporter = nodemailer.createTransport({
        host: smtpConfig.host,
        port: smtpConfig.port,
        secure: smtpConfig.secure,
        auth: {
          user: smtpConfig.user,
          pass: smtpConfig.pass,
        },
        // tls: {
        //   rejectUnauthorized: true,
        // },
      });
      await transporter.sendMail({
        from: smtpConfig.from,
        to: email,
        subject: 'Login OTP',
        html: `
              <!DOCTYPE html>
      <html lang="en">
        <head>
          <meta charset="utf-8" />
          <meta name="viewport" content="width=device-width, initial-scale=1" />
          <title>OTP Email Template</title>
          <link
            href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
            rel="stylesheet"
            integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH"
            crossorigin="anonymous"
          />
          <link
            rel="stylesheet"
            href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css"
          />

          <link rel="stylesheet" href="/style.css" />
          <style>
            body {
              font-family: Arial, sans-serif;
              background-color: #f4f4f4;
              padding: 0;
              margin: 0;
            }
            .container-sec {
              background-color: #ffffff;
              border-radius: 8px;
              padding: 20px;
              margin-top: 30px;
              box-shadow: 0px 0px 10px rgba(0, 0, 0, 0.1);
              max-width: 600px;
            }
            .otp-code {
              font-size: 24px;
              font-weight: bold;
              background-color: #f8f9fa;
              padding: 15px;
              text-align: center;
              border-radius: 8px;
              border: 1px dashed #007bff;
              color: #007bff;
            }
            .btn-verify {
              display: inline-block;
              padding: 10px 20px;
              color: #ffffff;
              background-color: #007bff;
              border-radius: 6px;
              text-decoration: none;
              font-weight: bold;
            }
            .footer-text {
              color: #6c757d;
              font-size: 14px;
              text-align: center;
              margin-top: 20px;
            }
            .footer-text a {
              color: #007bff;
              text-decoration: none;
            }
            .otp-lock {
              color: #333;
              font-size: 80px;
            }
            .welcome-section {
              background: #144fa9db;
              padding: 30px;
              border-radius: 4px;
              color: #fff;
              font-size: 20px;
              margin: 20px 0px;
            }
            .welcome-text {
              font-family: monospace;
            }
            .app-name {
              font-size: 30px;
              font-weight: 800;
              margin: 7px 0px;
            }
            .verify-text {
              margin-top: 25px;
              font-size: 25px;
              letter-spacing: 3px;
            }
            i.fas.fa-envelope-open {
              font-size: 35px !important;
              color: #ffffff;
            }
          </style>
        </head>

        <body>
          <div class="container-sec">
            <div class="text-center">
              <div class="welcome-section">
                <div class="verify-text">Please Verify Your Email Address</div>
                <div class="email-icon">
                  <i class="fas fa-envelope-open"></i>
                </div>
              </div>
              <h2>Dear Customer</h2>
              <p>Your One-Time Password (OTP) for verification is:</p>
              <div class="otp-code">${otp}</div>
              <p class="mt-4">
                Please use this OTP to complete your verification. The OTP is valid
                for the next 10 minutes.
              </p>
            </div>
            <div class="footer-text">
              <p>Thank you,<br />Privilege Team</p>
            </div>
          </div>

          <!-- <script
            src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"
            integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz"
            crossorigin="anonymous"
          ></script> -->
        </body>
      </html>

            `,
      });
      return true;
    } catch (error) {
      console.log(error);
      return false;
    }
  }

  async sendOtpInEmailForMaster(email, otp, adminAccId) {
    if (!email || !adminAccId) {
      return false;
    }
    try {
      const smtpConfig = await this.getSmtpConfigOfBusiness(adminAccId);
      const transporter = nodemailer.createTransport({
        host: smtpConfig.host,
        port: smtpConfig.port,
        secure: smtpConfig.secure,
        auth: {
          user: smtpConfig.user,
          pass: smtpConfig.pass,
        },
        // tls: {
        //   rejectUnauthorized: true,
        // },
      });
      await transporter.sendMail({
        from: smtpConfig.from,
        to: email,
        subject: 'Email Verification OTP',
        html: `
          <!DOCTYPE html>
  <html lang="en">
    <head>
      <meta charset="utf-8" />
      <meta name="viewport" content="width=device-width, initial-scale=1" />
      <title>OTP Email Template</title>
      <link
        href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
        rel="stylesheet"
        integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH"
        crossorigin="anonymous"
      />
      <link
        rel="stylesheet"
        href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css"
      />

      <link rel="stylesheet" href="/style.css" />
      <style>
        body {
          font-family: Arial, sans-serif;
          background-color: #f4f4f4;
          padding: 0;
          margin: 0;
        }
        .container-sec {
          background-color: #ffffff;
          border-radius: 8px;
          padding: 20px;
          margin-top: 30px;
          box-shadow: 0px 0px 10px rgba(0, 0, 0, 0.1);
          max-width: 600px;
        }
        .otp-code {
          font-size: 24px;
          font-weight: bold;
          background-color: #f8f9fa;
          padding: 15px;
          text-align: center;
          border-radius: 8px;
          border: 1px dashed #007bff;
          color: #007bff;
        }
        .btn-verify {
          display: inline-block;
          padding: 10px 20px;
          color: #ffffff;
          background-color: #007bff;
          border-radius: 6px;
          text-decoration: none;
          font-weight: bold;
        }
        .footer-text {
          color: #6c757d;
          font-size: 14px;
          text-align: center;
          margin-top: 20px;
        }
        .footer-text a {
          color: #007bff;
          text-decoration: none;
        }
        .otp-lock {
          color: #333;
          font-size: 80px;
        }
        .welcome-section {
          background: #144fa9db;
          padding: 30px;
          border-radius: 4px;
          color: #fff;
          font-size: 20px;
          margin: 20px 0px;
        }
        .welcome-text {
          font-family: monospace;
        }
        .app-name {
          font-size: 30px;
          font-weight: 800;
          margin: 7px 0px;
        }
        .verify-text {
          margin-top: 25px;
          font-size: 25px;
          letter-spacing: 3px;
        }
        i.fas.fa-envelope-open {
          font-size: 35px !important;
          color: #ffffff;
        }
      </style>
    </head>

    <body>
      <div class="container-sec">
        <div class="text-center">
          <div class="welcome-section">
            <div class="verify-text">Please Verify Your Email Address</div>
            <div class="email-icon">
              <i class="fas fa-envelope-open"></i>
            </div>
          </div>
          <h2>Dear Customer</h2>
          <p>Your One-Time Password (OTP) for verification is:</p>
          <div class="otp-code">${otp}</div>
          <p class="mt-4">
            Please use this OTP to complete your verification. The OTP is valid
            for the next 10 minutes.
          </p>
        </div>
        <div class="footer-text">
          <p>Thank you,<br />Choicest Enterprises Ltd.</p>
        </div>
      </div>

      <!-- <script
        src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"
        integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz"
        crossorigin="anonymous"
      ></script> -->
    </body>
  </html>`,
      });
      return true;
    } catch (error) {
      console.log(error);
      return false;
    }
  }

  async sendActivationMailFromMaster(email, adminAccId) {
    if (!email || !adminAccId) {
      return false;
    }
    try {
      const smtpConfig = await this.getSmtpConfigOfBusiness(adminAccId);
      const transporter = nodemailer.createTransport({
        host: smtpConfig.host,
        port: smtpConfig.port,
        secure: smtpConfig.secure,
        auth: {
          user: smtpConfig.user,
          pass: smtpConfig.pass,
        },
        // tls: {
        //   rejectUnauthorized: true,
        // },
      });
      await transporter.sendMail({
        from: smtpConfig.from,
        to: email,
        subject: 'Successful Software Activation',
        html: `<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Payment Successful - Privilege Membership</title>
  <style>
    body {
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
      background-color: #ffffff;
      color: #333333;
      margin: 0;
      padding: 0;
    }
    .container {
      max-width: 600px;
      margin: 30px auto;
      padding: 20px;
      border: 1px solid #e0e0e0;
      border-radius: 10px;
      background-color: #ffffff;
    }
    .logo {
      text-align: center;
      margin-bottom: 20px;
    }
    .logo img {
      max-height: 60px;
    }
    h1 {
      color: #2c3e50;
      font-size: 24px;
    }
    p {
      font-size: 16px;
      line-height: 1.6;
    }
    .footer {
      font-size: 12px;
      color: #888;
      text-align: center;
      margin-top: 30px;
    }
    .footer a {
      color: #888;
      text-decoration: none;
    }
  </style>
</head>
<body>
  <div class="container">
    <!-- <div class="logo">
      <img src="Logo.png" alt="Privilege Logo">
    </div> -->
    <h1>Software Activation Successful!</h1>
    <p>Dear User,</p>
    <p>We have successfully received your software activation request.</p>
    <p>Your admin dashboard will be activated shortly.</p>
    <p>Thank you for choosing us. We're excited to have you with us!</p>
    <p>Sincerely,<br><strong>The Webapps Software Solutions Team</strong></p>
    <div class="footer">
      <p>Webapps Software Solutions, 128, Jodhpur Gardens, Lake Gardens, Kolkata, West Bengal 700045</p>
      <p>Email: support@privilege.com | Phone: +123-456-7890</p>
      <p><a href="#">Unsubscribe</a></p>
    </div>
  </div>
</body>
</html>
`,
      });
      return true;
    } catch (error) {
      console.log(error);
      return false;
    }
  }

  async sendDeactivationMailFromMaster(email, adminAccId) {
    if (!email || !adminAccId) {
      return false;
    }
    try {
      const smtpConfig = await this.getSmtpConfigOfBusiness(adminAccId);
      const transporter = nodemailer.createTransport({
        host: smtpConfig.host,
        port: smtpConfig.port,
        secure: smtpConfig.secure,
        auth: {
          user: smtpConfig.user,
          pass: smtpConfig.pass,
        },
        // tls: {
        //   rejectUnauthorized: true,
        // },
      });
      await transporter.sendMail({
        from: smtpConfig.from,
        to: email,
        subject: 'Software Deactivated!',
        html: `<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Payment Successful - Privilege Membership</title>
  <style>
    body {
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
      background-color: #ffffff;
      color: #333333;
      margin: 0;
      padding: 0;
    }
    .container {
      max-width: 600px;
      margin: 30px auto;
      padding: 20px;
      border: 1px solid #e0e0e0;
      border-radius: 10px;
      background-color: #ffffff;
    }
    .logo {
      text-align: center;
      margin-bottom: 20px;
    }
    .logo img {
      max-height: 60px;
    }
    h1 {
      color: #2c3e50;
      font-size: 24px;
    }
    p {
      font-size: 16px;
      line-height: 1.6;
    }
    .footer {
      font-size: 12px;
      color: #888;
      text-align: center;
      margin-top: 30px;
    }
    .footer a {
      color: #888;
      text-decoration: none;
    }
  </style>
</head>
<body>
  <div class="container">
    <!-- <div class="logo">
      <img src="Logo.png" alt="Privilege Logo">
    </div> -->
    <h1>Software Deactivated!</h1>
    <p>Dear User,</p>
    <p>Unfortunately your software has been deactivated. Kindly contact with Admin.</p>
    <p>Thank you for choosing us. We're excited to have you with us!</p>
    <p>Sincerely,<br><strong>The Webapps Software Solutions Team</strong></p>
    <div class="footer">
      <p>Webapps Software Solutions, 128, Jodhpur Gardens, Lake Gardens, Kolkata, West Bengal 700045</p>
      <p>Email: support@privilege.com | Phone: +123-456-7890</p>
      <p><a href="#">Unsubscribe</a></p>
    </div>
  </div>
</body>
</html>
`,
      });
      return true;
    } catch (error) {
      console.log(error);
      return false;
    }
  }

  private async getSmtpConfigOfBusiness(businessAccId: string) {
    const emailCred = await this.busiEmailRepo.findOne({
      where: { accountId: businessAccId, status: DefaultStatus.ACTIVE },
    });
    return {
      host: emailCred.smtpHost,
      port: emailCred.smtpPort,
      secure: emailCred.secure,
      user: emailCred.username,
      pass: emailCred.password,
      from: emailCred.from,
    };
  }
}
