import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  UseGuards,
  Query,
  Put,
  FileTypeValidator,
  MaxFileSizeValidator,
  ParseFilePipe,
  UploadedFile,
  UseInterceptors,
} from '@nestjs/common';
import { MembershipCardService } from './membership-card.service';
import {
  CreateMembershipCardDto,
  MembershipCardPaginationDto,
} from './dto/create-membership-card.dto';
import { UpdateMembershipCardDto } from './dto/update-membership-card.dto';
import { AuthGuard } from '@nestjs/passport';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PermissionAction, UserRole } from 'src/enum';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Account } from 'src/account/entities/account.entity';
import { DefaultStatusDto } from 'src/common/dto/default-status.dto';
import { FileInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { extname } from 'path';
import { CommonPaginationDto } from 'src/common/dto/common-pagination.dto';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';

@Controller('membership-card')
export class MembershipCardController {
  constructor(private readonly membershipCardService: MembershipCardService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.CREATE, 'membership_card'])
  create(@Body() dto: CreateMembershipCardDto, @CurrentUser() user: Account) {
    dto.accountId = user.id;
    return this.membershipCardService.create(dto, user.id);
  }

  @Get('list')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'membership_card'])
  findAll(
    @Query() dto: MembershipCardPaginationDto,
    @CurrentUser() user: Account,
  ) {
    return this.membershipCardService.findAll(dto, user.id);
  }

  @Get('user/list/:accountId')
  find(
    @Query() dto: CommonPaginationDto,
    @Param('accountId') accountId: string,
  ) {
    return this.membershipCardService.find(dto, accountId);
  }

  @Get('user/particular/:id')
  findOneByUser(@Param('id') id: string) {
    return this.membershipCardService.findOneByUser(id);
  }

  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'membership_card'])
  update(@Param('id') id: string, @Body() dto: UpdateMembershipCardDto) {
    return this.membershipCardService.update(id, dto);
  }

  @Put('cardDesign/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard,PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'membership_card'])
  @UseInterceptors(
    FileInterceptor('file', {
      storage: diskStorage({
        destination: './uploads/MembershipCard',
        filename: (req, file, callback) => {
          const randomName = Array(32)
            .fill(null)
            .map(() => Math.round(Math.random() * 16).toString(16))
            .join('');
          return callback(null, `${randomName}${extname(file.originalname)}`);
        },
      }),
    }),
  )
  async cardDesign(
    @Param('id') id: string,
    @UploadedFile(
      new ParseFilePipe({
        validators: [
          new FileTypeValidator({ fileType: '.(png|jpeg|jpg)' }),
          new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 }),
        ],
      }),
    )
    file: Express.Multer.File,
  ) {
    const fileData = await this.membershipCardService.findOne(id);
    return this.membershipCardService.cardDesign(file.path, fileData);
  }

  @Put('status/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'membership_card'])
  status(@Param('id') id: string, @Body() dto: DefaultStatusDto) {
    return this.membershipCardService.status(id, dto);
  }
}
