import {
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import {
  CreateBusinessPaymentRecordDto,
  RecordPaginationDto,
} from './dto/create-business-payment-record.dto';
import { UpdateBusinessPaymentRecordDto } from './dto/update-business-payment-record.dto';
import { BusinessPaymentRecord } from './entities/business-payment-record.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Repository } from 'typeorm';
import { Business } from 'src/business/entities/business.entity';
import { Tax } from 'src/tax/entities/tax.entity';
import { DefaultStatus } from 'src/enum';

@Injectable()
export class BusinessPaymentRecordService {
  constructor(
    @InjectRepository(BusinessPaymentRecord)
    private readonly repo: Repository<BusinessPaymentRecord>,
    @InjectRepository(Business)
    private readonly businessRepo: Repository<Business>,
    @InjectRepository(Tax)
    private readonly taxRepo: Repository<Tax>,
  ) {}

  async create(dto: CreateBusinessPaymentRecordDto, masterAccId: string) {
    const result = await this.repo.findOne({
      where: { transactionId: dto.transactionId },
    });
    if (result) {
      throw new ConflictException(
        'Record already exists with this transactionId!',
      );
    }
    const business = await this.businessRepo.findOne({
      where: { accountId: dto.accountId },
    });
    if (!business) {
      throw new NotFoundException('Business Not Found With This ID!!');
    }

    const today = new Date();
    const datePart = today.toISOString().split('T')[0].replace(/-/g, '');
    const randomFourDigit = Math.floor(1000 + Math.random() * 9000);
    const invoiceNumber = `INV-${datePart}${randomFourDigit}`;

    const tax = await this.taxRepo.find({ where: { accountId: masterAccId, status: DefaultStatus.ACTIVE } });
    var total = dto.subTotal;
    if (tax.length > 0) {
      tax.map((t) => {
        var totalAmount = dto.subTotal * (parseFloat(t.rate) / 100);
        total += totalAmount;
      });
    }
    const due = total - dto.paid;
    

    dto.invoiceNumber = invoiceNumber;
    dto.businessName = business.businessName;
    dto.total = total;
    dto.amountDue = due;

    const obj = Object.assign(dto);
    return this.repo.save(obj);
  }

  async findAll(dto: RecordPaginationDto) {
    const fromDate = new Date(dto.fromDate);
    fromDate.setHours(0, 0, 0, 0);

    const toDate = new Date(dto.toDate);
    toDate.setHours(23, 59, 59, 999);

    const keyword = dto.keyword || '';
    const query = await this.repo.createQueryBuilder('businessPaymentRecord');
    if (dto.paymentMode && dto.paymentMode.length > 0) {
      query.andWhere('businessPaymentRecord.paymentMode = :paymentMode', {
        paymentMode: dto.paymentMode,
      });
    }
    if (dto.fromDate && dto.toDate) {
      query.andWhere(
        'businessPaymentRecord.paymentDate >= :fromDate AND businessPaymentRecord.paymentDate <= :toDate',
        {
          fromDate: fromDate,
          toDate: toDate,
        },
      );
    }
    if (dto.accountId && dto.accountId.length > 0) {
      query.andWhere('businessPaymentRecord.accountId = :accountId', {
        accountId: dto.accountId,
      });
    }
    if (dto.keyword && dto.keyword.length > 0) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where(
            'businessPaymentRecord.transactionId LIKE :keyword OR businessPaymentRecord.invoiceNumber LIKE :keyword OR businessPaymentRecord.subject LIKE :keyword OR businessPaymentRecord.businessName LIKE :keyword OR businessPaymentRecord.paid LIKE :keyword OR businessPaymentRecord.amountDue LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      );
    }

    const [result, total] = await query
      .orderBy({ 'businessPaymentRecord.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async findDetail(id: string) {
    const result = await this.repo
      .createQueryBuilder('businessPaymentRecord')
      .leftJoinAndSelect('businessPaymentRecord.account', 'account')
      .leftJoinAndSelect('account.tax', 'tax')
      .leftJoinAndSelect('account.business', 'business')
      .select([
        'businessPaymentRecord.id',
        'businessPaymentRecord.accountId',
        'businessPaymentRecord.subject',
        'businessPaymentRecord.invoiceNumber',
        'businessPaymentRecord.transactionId',
        'businessPaymentRecord.businessName',
        'businessPaymentRecord.subTotal',
        'businessPaymentRecord.total',
        'businessPaymentRecord.paid',
        'businessPaymentRecord.amountDue',
        'businessPaymentRecord.paymentMode',
        'businessPaymentRecord.paymentDate',
        'businessPaymentRecord.note',
        'businessPaymentRecord.createdAt',

        'account.id',

        'tax.id',
        'tax.taxName',
        'tax.rate',

        'business.id',
        'business.personName',
        'business.personDesignation',
        'business.gstNo',
      ])
      .where('businessPaymentRecord.id = :id', { id: id })
      .getOne();

    if (!result) {
      throw new NotFoundException('Record not found!');
    }

    return result;
  }

  async findOne(id: string) {
    const result = await this.repo
      .createQueryBuilder('businessPaymentRecord')
      .leftJoinAndSelect('businessPaymentRecord.account', 'account')
      .leftJoinAndSelect('account.tax', 'tax')
      .leftJoinAndSelect('account.business', 'business')
      .select([
        'businessPaymentRecord.id',
        'businessPaymentRecord.accountId',
        'businessPaymentRecord.subject',
        'businessPaymentRecord.invoiceNumber',
        'businessPaymentRecord.transactionId',
        'businessPaymentRecord.businessName',
        'businessPaymentRecord.subTotal',
        'businessPaymentRecord.total',
        'businessPaymentRecord.paid',
        'businessPaymentRecord.amountDue',
        'businessPaymentRecord.paymentMode',
        'businessPaymentRecord.paymentDate',
        'businessPaymentRecord.note',
        'businessPaymentRecord.createdAt',

        'account.id',

        'tax.id',
        'tax.taxName',
        'tax.rate',

        'business.id',
        'business.personName',
        'business.personDesignation',
        'business.gstNo',
      ])
      .where('businessPaymentRecord.id = :id', { id: id })
      .getOne();

    if (!result) {
      throw new NotFoundException('Record not found!');
    }

    return result;
  }

  async update(id: string, dto: UpdateBusinessPaymentRecordDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Record not found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async remove(id: string) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Record not found!');
    }
    return this.repo.remove(result);
  }
}
