import { ApiProperty } from '@nestjs/swagger';
import { Type } from 'class-transformer';
import {
  IsDate,
  IsEmail,
  IsEnum,
  IsNotEmpty,
  IsNumber,
  IsOptional,
  IsString,
  Max,
  MaxLength,
  Min,
  MinLength,
} from 'class-validator';
import {
  BillTo,
  DefaultStatus,
  Gender,
  MemberStatus,
  StaffRole,
  SubscriptionStatus,
  UserRole,
  YNStatus,
} from 'src/enum';

export class CreateAccountDto {
  @ApiProperty()
  @IsNotEmpty()
  @MinLength(1)
  @MaxLength(55)
  email: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsString()
  @MinLength(5)
  @MaxLength(30)
  password: string;

  @ApiProperty()
  @IsNotEmpty()
  salutation: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsEnum(Gender)
  gender: Gender;

  @ApiProperty()
  @IsNotEmpty()
  @IsString()
  @MinLength(2)
  @MaxLength(30)
  fName: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsString()
  @MinLength(2)
  @MaxLength(30)
  lName: string;

  @ApiProperty()
  @IsNotEmpty()
  @MinLength(10)
  @MaxLength(10)
  mobile: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsEnum(StaffRole)
  staffRole: StaffRole;

  @ApiProperty()
  @IsNotEmpty()
  @IsEnum(DefaultStatus)
  status: DefaultStatus;
}

export class UpdateStaffPasswordDto {
  @IsOptional()
  @IsString()
  loginId: string;

  @IsOptional()
  password: string;
}

export class UpdateStaffDto {
  @ApiProperty()
  @IsOptional()
  salutation: string;

  @ApiProperty()
  @IsOptional()
  gender: Gender;

  @ApiProperty()
  @IsOptional()
  @IsString()
  fName: string;

  @ApiProperty()
  @IsOptional()
  @IsString()
  lName: string;

  @ApiProperty()
  @IsOptional()
  mobile: string;

  @ApiProperty()
  @IsOptional()
  staffRole: StaffRole;
}

export class PaginationDto {
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(10)
  @Max(50)
  limit: number;

  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  @Max(1000)
  offset: number;

  @IsOptional()
  @MinLength(0)
  @MaxLength(100)
  keyword: string;

  @IsOptional()
  @IsEnum(DefaultStatus)
  status: DefaultStatus;
}

export class StatusDto {
  @IsNotEmpty()
  @IsEnum(DefaultStatus)
  status: DefaultStatus;
}

export class EmailUpdateDto {
  @IsOptional()
  @IsString()
  @MinLength(0)
  @MaxLength(50)
  email: string;

  @IsOptional()
  otp: string;
}

export class AddMemberDto {
  @IsNotEmpty()
  phoneNumber: string;

  @IsNotEmpty()
  salutation: string;

  @IsNotEmpty()
  email: string;

  @IsNotEmpty()
  fName: string;

  @IsOptional()
  mName: string;

  @IsNotEmpty()
  lName: string;

  @IsNotEmpty()
  gender: string;

  @IsNotEmpty()
  address1: string;

  @IsNotEmpty()
  address2: string;

  @IsNotEmpty()
  city: string;

  @IsNotEmpty()
  state: string;

  @IsNotEmpty()
  landMark: string;

  @IsNotEmpty()
  zipcode: string;

  @IsOptional()
  qualification: string;

  @IsOptional()
  aadharNumber: string;

  @IsOptional()
  panNumber: string;

  @IsNotEmpty()
  haveBusiness: YNStatus;

  @IsOptional()
  businessType: string;

  @IsOptional()
  businessName: string;

  @IsOptional()
  businessEmail: string;

  @IsOptional()
  gstNumber: string;

  @IsOptional()
  businessCity: string;

  @IsOptional()
  businessState: string;

  @IsOptional()
  businessZipcode: string;

  @IsOptional()
  businessPhone: string;

  @IsOptional()
  businessLandmark: string;

  @IsOptional()
  businessAddress1: string;

  @IsOptional()
  businessAddress2: string;

  @IsOptional()
  fatherName: string;

  @IsOptional()
  dob: string;

  @IsOptional()
  maritalStatus: string;

  @IsOptional()
  businessTurnover: string;

  @IsNotEmpty()
  @IsEnum(BillTo)
  billTo: BillTo;

  @IsOptional()
  @MinLength(0)
  @MaxLength(100)
  profession: string;

  @IsOptional()
  @MinLength(0)
  @MaxLength(100)
  income: string;

  @IsNotEmpty()
  membershipCardId: string;
}

export class MemberPaginationDto {
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(10)
  @Max(50)
  limit: number;

  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  @Max(1000)
  offset: number;

  @IsOptional()
  @MinLength(0)
  @MaxLength(100)
  keyword: string;

  @IsOptional()
  status: MemberStatus;

  @IsOptional()
  phoneNumber: string;

  @IsOptional()
  membershipType: string;

  @IsOptional()
  memberId: string;

  @IsOptional()
  startDate: string;

  @IsOptional()
  endDate: string;

  @IsOptional()
  joinStartDate: string;

  @IsOptional()
  joinEndDate: string;

  @IsOptional()
  paymentStatus: SubscriptionStatus;
}

export class CSVMemberPaginationDto {
  @IsOptional()
  @MinLength(0)
  @MaxLength(100)
  keyword: string;

  @IsOptional()
  status: MemberStatus;

  @IsOptional()
  phoneNumber: string;

  @IsOptional()
  membershipType: string;

  @IsOptional()
  memberId: string;

  @IsOptional()
  startDate: string;

  @IsOptional()
  endDate: string;

  @IsOptional()
  joinStartDate: string;

  @IsOptional()
  joinEndDate: string;

  @IsOptional()
  paymentStatus: SubscriptionStatus;
}

export class SearchMemberPaginationDto {
  @IsNotEmpty()
  keyword: string;
}

export class ChildQrVerifyDto {
  @IsNotEmpty()
  businessAccId: string;

  @IsNotEmpty()
  userChildId: string;
}

export class BusinessCreateByMasterDto {
  @IsNotEmpty()
  @MinLength(2)
  @MaxLength(100)
  email: string;

  @IsNotEmpty()
  @MinLength(10)
  @MaxLength(10)
  phoneNumber: string;

  @IsNotEmpty()
  @MinLength(8)
  password: string;
}
