import { BadRequestException, Injectable } from '@nestjs/common';
import { CreateUserGameDto } from './dto/create-user-game.dto';
import { UpdateUserGameDto } from './dto/update-user-game.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { UserGame } from './entities/user-game.entity';
import { Repository } from 'typeorm';
import { PlayGameStatus } from 'src/enum';
import { Account } from 'src/account/entities/account.entity';
import { Game } from 'src/game/entities/game.entity';
import { UserCategory } from 'src/user-category/entities/user-category.entity';

@Injectable()
export class UserGameService {
  constructor(
    @InjectRepository(UserGame) private readonly repo: Repository<UserGame>,
    @InjectRepository(UserCategory)
    private readonly ucRepo: Repository<UserCategory>,
    @InjectRepository(Account) private readonly accRepo: Repository<Account>,
    @InjectRepository(Game) private readonly gameRepo: Repository<Game>,
  ) {}

  async create(dto: CreateUserGameDto, accountId: string) {
    const findCategory = await this.gameRepo.findOne({
      where: { id: dto.gameId },
    });
    const result = await this.repo.findOne({
      where: { accountId, gameId: dto.gameId },
    });
    if (!result) {
      const obj = Object.assign({
        categoryId: findCategory.categoryId,
        accountId: accountId,
        gameId: dto.gameId,
        totalDuration: null,
        status: PlayGameStatus.PLAYING,
        attemptCount: null,
        correctCount: null,
      });
      return this.repo.save(obj);
    } else {
      if (!dto.totalDuration) {
        const obj = Object.assign(result, {
          // totalDuration: result.totalDuration,
          status: dto.status,
        });
        return this.repo.save(obj);
      }
      if (dto.totalDuration) {
        const obj = Object.assign(result, {
          totalDuration: result.totalDuration + dto.totalDuration,
          earnedPoint: result.earnedPoint + dto.earnedPoint,
          attemptCount: result.attemptCount + dto.attemptCount,
          correctCount: result.correctCount + dto.correctCount,
          status: dto.status,
          accuracy:
            ((result.correctCount + dto.correctCount) /
              (result.attemptCount + dto.attemptCount)) *
            100,
        });
        const ug = await this.repo.save(obj);
        const [findAllGames, total] = await this.repo.findAndCount({
          where: { accountId, categoryId: findCategory.categoryId },
        });
        let totalAccuracy = 0;
        let totalDuration = 0;
        findAllGames.forEach((item) => {
          totalAccuracy += item.accuracy;
          totalDuration += item.totalDuration;
        });
        const accuracy = totalAccuracy / total;
        const duration = totalDuration / total;
        const findRecord = await this.ucRepo.findOne({
          where: { accountId, categoryId: findCategory.categoryId },
        });
        findRecord.accuracy = accuracy;
        findRecord.avgDuration = duration;
        this.ucRepo.save(findRecord);

        return ug;
      }
    }
  }

  async findDetail(gameId: string, accountId: string) {
    const result = await this.repo.findOne({
      select: [
        'id',
        'totalDuration',
        'earnedPoint',
        'attemptCount',
        'createdAt',
        'status',
      ],
      where: { accountId, gameId },
    });
    if (!result) {
      throw new BadRequestException('Game not played yet!');
    }
    return result;
  }

  async viewResult(categoryId: string, accountId: string) {
    const result = await this.repo
      .createQueryBuilder('userGame')
      .leftJoinAndSelect('userGame.game', 'game')
      .select([
        'userGame.id',
        'userGame.totalDuration',
        'userGame.earnedPoint',
        'userGame.attemptCount',
        'userGame.createdAt',
        'userGame.status',

        'game.id',
        'game.gameName',
      ])
      .where(
        'userGame.categoryId = :categoryId AND userGame.accountId = :accountId',
        { categoryId: categoryId, accountId: accountId },
      )
      .getMany();
    if (!result) {
      throw new BadRequestException('Game not played yet!');
    }
    return { result };
  }
}
