import {
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import { CreatePlanDto, PlanPaginationDto } from './dto/create-plan.dto';
import { UpdatePlanDto } from './dto/update-plan.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { Plan } from './entities/plan.entity';
import { Brackets, Repository } from 'typeorm';
import { DefaultStatus } from 'src/enum';
import { DefaultStatusDto } from 'src/common/dto/default-status.dto';

@Injectable()
export class PlanService {
  constructor(
    @InjectRepository(Plan) private readonly repo: Repository<Plan>,
  ) {}

  async create(dto: CreatePlanDto) {
    const result = await this.repo.findOne({
      where: { planType: dto.planType, planName: dto.planName },
    });
    if (result) {
      throw new ConflictException('Plan already exists!');
    }
    const obj = Object.assign(dto);
    return this.repo.save(obj);
  }

  async findAll(dto: PlanPaginationDto) {
    const keyword = dto.keyword || '';
    const query = await this.repo.createQueryBuilder('plan');
    if (dto.status && dto.status.length > 0) {
      query.andWhere('plan.status = :status', { status: dto.status });
    }
    if (dto.planType && dto.planType.length > 0) {
      query.andWhere('plan.planType = :planType', { planType: dto.planType });
    }
    if (dto.keyword && dto.keyword.length > 0) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where('plan.planName LIKE :keyword OR plan.price LIKE :keyword', {
            keyword: '%' + keyword + '%',
          });
        }),
      );
    }
    const [result, total] = await query
      .orderBy({ 'plan.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async findByUser(dto: PlanPaginationDto) {
    const query = await this.repo
      .createQueryBuilder('plan')
      .where('plan.status = :status AND plan.planType = :planType', {
        status: DefaultStatus.ACTIVE,
        planType: dto.planType,
      });
    const [result, total] = await query
      .orderBy({ 'plan.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async popular() {
    const result = await this.repo.findOne({
      where: { popular: true, status: DefaultStatus.ACTIVE },
    });
    if (!result) {
      throw new NotFoundException('Plan not found!');
    }
    return result;
  }

  async findOne(id: string) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Plan not found!');
    }
    return result;
  }

  async update(id: string, dto: UpdatePlanDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Plan not found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async status(id: string, dto: DefaultStatusDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Plan not found!');
    }
    const obj = Object.assign(result, { status: dto.status });
    return this.repo.save(obj);
  }
}
