import { HttpService } from '@nestjs/axios';
import { Injectable, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { In, Repository } from 'typeorm';
import { Notification } from './entities/notification.entity';
import {
  BulkDeleteDto,
  NotificationDto,
  UpdateNotificationDto,
} from './dto/notification.dto';
import { unlink } from 'fs/promises';
import { join } from 'path';
import { Account } from 'src/account/entities/account.entity';
import { DefaultStatus, UserRole } from 'src/enum';

@Injectable()
export class NotificationsService {
  constructor(
    @InjectRepository(Notification)
    private readonly repo: Repository<Notification>,
    @InjectRepository(Account)
    private readonly accRepo: Repository<Account>,
    private readonly httpService: HttpService,
  ) {}

  async sendBulkNotification(body, title, token, status) {
    const header = {
      headers: {
        'cache-control': 'no-cache',
        authorization:
          'key=AAAAqKmoPwY:APA91bEuJpsVMvfhzcwPbXUV3B6Wu6kQl8iA6738dXuvdMHSELGZegyGLc90uP0LqTSGkzMv08ULzE29_lDsvJTSUr2BH2Flk-w2',
        'content-type': 'application/json',
      },
    };
    let data = null;
    if (status) {
      data = JSON.stringify({
        registration_ids: token,
        data: { body: body, title: title, sound: 'default', type: 1 },
        notification: { body: body, title: title, sound: 'default', type: 1 },
      });
    } else {
      data = JSON.stringify({
        to: token,
        data: { body: body, title: title, sound: 'default', type: 1 },
        notification: { body: body, title: title, sound: 'default', type: 1 },
      });
    }

    try {
      const result = await this.httpService.axiosRef.post(
        `https://fcm.googleapis.com/fcm/send`,
        data,
        header,
      );
      if (result.data) {
        return result.data;
      }
    } catch (error) {
      return false;
    }
  }

  // async create(createDto) {
  //   const result = await this.repo.count({
  //     where: {
  //       title: createDto.title,
  //       desc: createDto.desc,
  //       type: createDto.type,
  //       accountId: createDto.accountId,
  //     },
  //   });

  //   if (!result) {
  //     return this.repo.save(createDto);
  //   } else {
  //     return true;
  //   }
  // }

  async create(image: string, createDto: NotificationDto) {
    if (!createDto.accountId) {
      const accounts = await this.accRepo.find({
        where: { roles: UserRole.USER, status: DefaultStatus.ACTIVE },
      });
      if (!accounts) {
        throw new NotFoundException('No accounts found..');
      }
      accounts.map(async (acc) => {
        const result = await this.repo.findOne({
          where: {
            title: createDto.title,
            desc: createDto.desc,
            accountId: acc.id,
          },
        });
        if (!result) {
          this.repo.save({
            title: createDto.title,
            desc: createDto.desc,
            accountId: acc.id,
            image: !image ? null : process.env.CLU_CDN_LINK + image,
            imagePath: image,
          });
        }
      });
      return { sucess: true, message: 'Notification created for all users.' };
    } else {
      const result = await this.repo.count({
        where: {
          title: createDto.title,
          desc: createDto.desc,
          // type: createDto.type,
          accountId: createDto.accountId,
        },
      });
      if (!result) {
        return this.repo.save({
          title: createDto.title,
          desc: createDto.desc,
          accountId: createDto.accountId,
          image: !image ? null : process.env.CLU_CDN_LINK + image,
          imagePath: image,
        });
      } else {
        return true;
      }
    }
  }

  async findOne(id: number) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Notification Not Found..');
    }
    return result;
  }

  async findAll(limit: number, offset: number, accountId: string) {
    const [result, total] = await this.repo
      .createQueryBuilder('notification')
      .leftJoinAndSelect('notification.account', 'account')
      .where(
        'notification.accountId = :accountId OR notification.accountId IS NULL',
        {
          accountId: accountId,
        },
      )
      .skip(offset)
      .take(limit)
      .orderBy({ 'notification.createdAt': 'DESC' })
      .getManyAndCount();
    return { result, total };
  }

  async findAllByAdmin(limit: number, offset: number) {
    const [result, total] = await this.repo
      .createQueryBuilder('notification')
      .leftJoinAndSelect('notification.account', 'account')
      .skip(offset)
      .take(limit)
      .orderBy({ 'notification.createdAt': 'DESC' })
      .getManyAndCount();
    return { result, total };
  }

  async updateNotif(id: number, dto: UpdateNotificationDto) {
    const notifs = await this.repo.findOne({ where: { id } });
    if (!notifs) {
      throw new NotFoundException('Notification not found!');
    }
    const obj = Object.assign(notifs, dto);
    return this.repo.save(obj);
  }

  async update(id: number, accountId: string, status: boolean) {
    const notifs = await this.repo.findOne({ where: { id, accountId } });
    if (!notifs) {
      throw new NotFoundException('Notification not found!');
    }
    const obj = Object.assign(notifs, { read: status });
    return this.repo.save(obj);
  }

  async readNoti(id: number) {
    const notifs = await this.repo.findOne({ where: { id } });
    if (!notifs) {
      throw new NotFoundException('Notification not found!');
    }
    const obj = Object.assign(notifs, { read: true });
    return this.repo.save(obj);
  }

  async image(image: string, result: Notification) {
    if (result.imagePath) {
      const oldPath = join(__dirname, '..', '..', result.imagePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      image: process.env.CLU_CDN_LINK + image,
      imagePath: image,
    });
    return this.repo.save(obj);
  }

  async remove(id: number) {
    const notifs = await this.repo.findOne({ where: { id } });
    if (!notifs) {
      throw new NotFoundException('Notification not found!');
    }
    return this.repo.remove(notifs);
  }

  async bulkDelete(dto: BulkDeleteDto) {
    const notifications = await this.repo.findBy({
      id: In(dto.ids),
    });

    if (!notifications.length) {
      throw new NotFoundException('No notifications found!');
    }

    await this.repo.remove(notifications);

    return {
      success: true,
      message: 'All notifications deleted successfully.',
    };
  }
}
