import {
  BadRequestException,
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import { CreateGameDto, GamePaginationDto } from './dto/create-game.dto';
import { UpdateGameDto } from './dto/update-game.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { Game } from './entities/game.entity';
import { Brackets, Repository } from 'typeorm';
import { DefaultStatusDto } from 'src/common/dto/default-status.dto';
import { unlink } from 'fs/promises';
import { join } from 'path';
import { CategoryPublishStatus, DefaultStatus, GameType } from 'src/enum';

@Injectable()
export class GameService {
  constructor(
    @InjectRepository(Game) private readonly repo: Repository<Game>,
  ) {}

  async create(dto: CreateGameDto) {
    const result = await this.repo.findOne({
      where: { categoryId: dto.categoryId, gameName: dto.gameName },
    });
    if (result) {
      throw new ConflictException('Game already exists!');
    }
    const obj = Object.assign({
      categoryId: dto.categoryId,
      gameName: dto.gameName,
      gameDesc: dto.gameDesc,
      resultTimer: dto.resultTimer,
      type: dto.type,
      pubStatus:
        dto.type == GameType.C3
          ? (dto.pubStatus = CategoryPublishStatus.UNPUBLISH)
          : (dto.pubStatus = CategoryPublishStatus.PUBLISH),
    });
    return this.repo.save(obj);
  }

  async findAll(dto: GamePaginationDto) {
    const keyword = dto.keyword || '';
    const query = await this.repo.createQueryBuilder('game');
    if (dto.status && dto.status.length > 0) {
      query.andWhere('game.status = :status', { status: dto.status });
    }
    if (dto.pubStatus && dto.pubStatus.length > 0) {
      query.andWhere('game.pubStatus = :pubStatus', {
        pubStatus: dto.pubStatus,
      });
    }
    if (dto.categoryId && dto.categoryId.length > 0) {
      query.andWhere('game.categoryId = :categoryId', {
        categoryId: dto.categoryId,
      });
    }
    if (dto.keyword && dto.keyword.length > 0) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where(
            'game.gameName LIKE :keyword OR game.gameDesc LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      );
    }
    const [result, total] = await query
      .orderBy({ 'game.type': 'ASC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async findByUser(categoryId: string) {
    const query = await this.repo
      .createQueryBuilder('game')
      .leftJoinAndSelect('game.userGame', 'userGame')
      .select([
        'game.id',
        'game.categoryId',
        'game.gameName',
        'game.gameDesc',
        'game.gameImage',
        'game.gameIcon',
        'game.publishDateFrom',
        'game.publishDateTo',
        'game.gameOpenTimer',
        'game.gameCloseTime',
        'game.resultTimer',
        'game.type',
        'game.status',
        'game.createdAt',

        'userGame.id',
        'userGame.accountId',
        'userGame.attemptCount',
        'userGame.status',
      ])
      .where('game.status = :status AND game.categoryId = :categoryId', {
        status: DefaultStatus.ACTIVE,
        categoryId: categoryId,
      });
    const [result, total] = await query
      // .orderBy({ 'game.createdAt': 'DESC' })
      .getManyAndCount();

    return { result, total };
  }

  async findOne(id: string) {
    const result = await this.repo.findOne({ where: { id: id } });
    if (!result) {
      throw new NotFoundException('Game Not Found..');
    }
    return result;
  }

  async update(id: string, dto: UpdateGameDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Game not found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async image(image: string, result: Game) {
    if (result.gameImagePath) {
      const oldPath = join(__dirname, '..', '..', result.gameImagePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      gameImage: process.env.CLU_CDN_LINK + image,
      gameImagePath: image,
    });
    return this.repo.save(obj);
  }

  async gameIcon(image: string, result: Game) {
    if (result.gameIconPath) {
      const oldPath = join(__dirname, '..', '..', result.gameIconPath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      gameIcon: process.env.CLU_CDN_LINK + image,
      gameIconPath: image,
    });
    return this.repo.save(obj);
  }

  async status(id: string, dto: DefaultStatusDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Game not found!');
    }
    const obj = Object.assign(result, { status: dto.status });
    return this.repo.save(obj);
  }
}
