import { Injectable, Logger } from '@nestjs/common';
import { Cron, CronExpression } from '@nestjs/schedule';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { CategoryPublishStatus, DefaultStatus } from 'src/enum';
import { Game } from './entities/game.entity';

@Injectable()
export class GameCronService {
  private readonly logger = new Logger(GameCronService.name);

  constructor(
    @InjectRepository(Game)
    private readonly gameRepo: Repository<Game>,
  ) {}

  /**********For Local*********/
  // @Cron(CronExpression.EVERY_MINUTE)
  // async handleQuickFirePublish() {
  //   const now = new Date();
  //   const currentDate = now.toISOString().split('T')[0]; // YYYY-MM-DD
  //   const currentTime = now.toTimeString().slice(0, 5); // HH:mm

  //   // Find QF games to PUBLISH (time window reached)
  //   const toPublish = await this.gameRepo
  //     .createQueryBuilder('game')
  //     .where('game.publishDateFrom = :date', { date: currentDate })
  //     .andWhere('game.gameOpenTimer <= :time', { time: currentTime })
  //     .andWhere('game.status = :status', { status: DefaultStatus.ACTIVE })
  //     .andWhere('game.pubStatus = :pubStatus', {
  //       pubStatus: CategoryPublishStatus.UNPUBLISH,
  //     })
  //     .getMany();

  //   if (toPublish.length == 0) {
  //     this.logger.log(`No Games found to Publish!`);
  //   }
  //   for (const game of toPublish) {
  //     game.pubStatus = CategoryPublishStatus.PUBLISH;
  //     await this.gameRepo.save(game);
  //     this.logger.log(`Published: ${game.gameName}`);
  //   }

  //   // Find QF games to UNPUBLISH (close time reached)
  //   const toUnpublish = await this.gameRepo
  //     .createQueryBuilder('game')
  //     .where('game.publishDateTo = :date', { date: currentDate })
  //     .andWhere('game.gameCloseTime <= :time', { time: currentTime })
  //     .andWhere('game.status = :status', { status: DefaultStatus.ACTIVE })
  //     .andWhere('game.pubStatus = :pubStatus', {
  //       pubStatus: CategoryPublishStatus.PUBLISH,
  //     })
  //     .getMany();

  //   if (toUnpublish.length == 0) {
  //     this.logger.log(`No games found to Unpublish!`);
  //   }
  //   for (const game of toUnpublish) {
  //     game.pubStatus = CategoryPublishStatus.UNPUBLISH;
  //     await this.gameRepo.save(game);
  //     this.logger.log(`Unpublished: ${game.gameName}`);
  //   }
  // }

  /**********For Server*********/
  @Cron(CronExpression.EVERY_MINUTE)
  async handleQuickFirePublish() {
    const now = new Date();
    const later = new Date(now.getTime() + (4 * 60 + 30) * 60000); // 4h 30m in ms

    const currentDate = later.toISOString().split('T')[0];
    const currentTime = later.toTimeString().slice(0, 5);

    // Find QF games to PUBLISH (time window reached)
    const toPublish = await this.gameRepo
      .createQueryBuilder('game')
      .where('game.publishDateFrom = :date', { date: currentDate })
      .andWhere('game.gameOpenTimer <= :time', { time: currentTime })
      .andWhere('game.status = :status', { status: DefaultStatus.ACTIVE })
      .andWhere('game.pubStatus = :pubStatus', {
        pubStatus: CategoryPublishStatus.UNPUBLISH,
      })
      .getMany();

    if (toPublish.length == 0) {
      this.logger.log(`No Games found to Publish!`);
    }
    for (const game of toPublish) {
      game.pubStatus = CategoryPublishStatus.PUBLISH;
      await this.gameRepo.save(game);
      this.logger.log(`Published: ${game.gameName}`);
    }

    // Find QF games to UNPUBLISH (close time reached)
    const toUnpublish = await this.gameRepo
      .createQueryBuilder('game')
      .where('game.publishDateTo = :date', { date: currentDate })
      .andWhere('game.gameCloseTime <= :time', { time: currentTime })
      .andWhere('game.status = :status', { status: DefaultStatus.ACTIVE })
      .andWhere('game.pubStatus = :pubStatus', {
        pubStatus: CategoryPublishStatus.PUBLISH,
      })
      .getMany();

    if (toUnpublish.length == 0) {
      this.logger.log(`No games found to Unpublish!`);
    }
    for (const game of toUnpublish) {
      game.pubStatus = CategoryPublishStatus.UNPUBLISH;
      await this.gameRepo.save(game);
      this.logger.log(`Unpublished: ${game.gameName}`);
    }
  }
}
