import {
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import {
  CreateFeedbackDto,
  FeedbackPaginationDto,
  FeedbackStatusDto,
} from './dto/create-feedback.dto';
import { UpdateFeedbackDto } from './dto/update-feedback.dto';
import { Feedback } from './entities/feedback.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Like, Repository } from 'typeorm';
import { CommonPaginationDto } from 'src/common/dto/common-pagination.dto';
import { FeedbackStatus } from 'src/enum';

@Injectable()
export class FeedbackService {
  constructor(
    @InjectRepository(Feedback) private readonly repo: Repository<Feedback>,
  ) {}

  async create(dto: CreateFeedbackDto) {
    try {
      const obj = Object.create(dto);
      return this.repo.save(obj);
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async findAll(dto: FeedbackPaginationDto) {
    try {
      const keyword = dto.keyword || '';
      const query = this.repo
        .createQueryBuilder('feedback')
        .leftJoinAndSelect('feedback.account', 'account')
        .select([
          'feedback.id',
          'feedback.moduleName',
          'feedback.issueName',
          'feedback.suggestion',
          'feedback.status',
          'feedback.createdAt',

          'account.id',
          'account.phoneNumber',
        ])
        .where('feedback.status = :status', { status: dto.status });
      if (dto.keyword && dto.keyword.length > 0) {
        query.andWhere(
          new Brackets((qb) => {
            qb.where(
              'account.phoneNumber LIKE :keyword OR feedback.moduleName LIKE :keyword OR feedback.issueName LIKE :keyword OR feedback.suggestion LIKE :keyword ',
              {
                keyword: '%' + keyword + '%',
              },
            );
          }),
        );
      }
      const [result, total] = await query
        .orderBy('feedback.createdAt', 'DESC')
        .take(dto.limit)
        .skip(dto.offset)
        .getManyAndCount();
      return { result, total };
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async find(dto: CommonPaginationDto, accountId: string) {
    try {
      const [result, total] = await this.repo.findAndCount({
        where: { accountId: accountId },
        order: { createdAt: 'DESC' },
        take: dto.limit,
        skip: dto.offset,
      });
      return { result, total };
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async statusByUser(id: string) {
    try {
      const result = await this.repo.findOne({ where: { id } });
      if (!result) {
        throw new NotFoundException('Feedback not found!');
      }
      const obj = Object.assign(result, { status: FeedbackStatus.RESOLVED });
      return this.repo.save(obj);
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async status(id: string, dto: FeedbackStatusDto) {
    try {
      const result = await this.repo.findOne({ where: { id } });
      if (!result) {
        throw new NotFoundException('Feedback not found!');
      }
      const obj = Object.assign(result, { status: dto.status });
      return this.repo.save(obj);
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async remove(id: string) {
    try {
      const result = await this.repo.findOne({ where: { id } });
      if (!result) {
        throw new NotFoundException('Feedback not found!');
      }
      return this.repo.remove(result);
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }
}
