import {
  ConflictException,
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import { CreateFeedbackIssueDto } from './dto/create-feedback-issue.dto';
import { UpdateFeedbackIssueDto } from './dto/update-feedback-issue.dto';
import { FeedbackIssue } from './entities/feedback-issue.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Like, Repository } from 'typeorm';
import { CommonPaginationDto } from 'src/common/dto/common-pagination.dto';

@Injectable()
export class FeedbackIssueService {
  constructor(
    @InjectRepository(FeedbackIssue)
    private readonly repo: Repository<FeedbackIssue>,
  ) {}

  async create(dto: CreateFeedbackIssueDto) {
    try {
      const result = await this.repo.findOne({
        where: { issueName: dto.issueName },
      });
      if (result) {
        throw new ConflictException('Issue Name already exists!');
      }
      const obj = Object.assign(dto);
      return this.repo.save(obj);
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async findAll(dto: CommonPaginationDto) {
    try {
      const keyword = dto.keyword || '';
      const [result, total] = await this.repo.findAndCount({
        where: {
          issueName: Like('%' + keyword + '%'),
        },
        order: { createdAt: 'DESC' },
        take: dto.limit,
        skip: dto.offset,
      });
      return { result, total };
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async find() {
    try {
      const [result, total] = await this.repo.findAndCount({
        select: ['id', 'issueName', 'createdAt'],
        order: { createdAt: 'DESC' },
      });
      return { result, total };
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async update(id: string, dto: CreateFeedbackIssueDto) {
    try {
      const result = await this.repo.findOne({ where: { id } });
      if (!result) {
        throw new NotFoundException('Issue Name not found!');
      }
      const obj = Object.assign(result, dto);
      return this.repo.save(obj);
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async remove(id: string) {
    try {
      const result = await this.repo.findOne({ where: { id } });
      if (!result) {
        throw new NotFoundException('Issue Name not found!');
      }
      return this.repo.remove(result);
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }
}
