import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  UseGuards,
  Query,
} from '@nestjs/common';
import { CoinHistoryService } from './coin-history.service';
import { CreateCoinHistoryDto } from './dto/create-coin-history.dto';
import { UpdateCoinHistoryDto } from './dto/update-coin-history.dto';
import { AuthGuard } from '@nestjs/passport';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { UserRole } from 'src/enum';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Account } from 'src/account/entities/account.entity';
import { CommonPaginationDto } from 'src/common/dto/common-pagination.dto';

@Controller('coin-history')
export class CoinHistoryController {
  constructor(private readonly coinHistoryService: CoinHistoryService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  create(@Body() dto: CreateCoinHistoryDto, @CurrentUser() user: Account) {
    dto.accountId = user.id;
    return this.coinHistoryService.create(dto);
  }

  @Get('history')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  findAll(@CurrentUser() user: Account, @Query() dto: CommonPaginationDto) {
    return this.coinHistoryService.findAll(user.id, dto);
  }

  @Get('list/:accountId')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN)
  find(
    @Param('accountId') accountId: string,
    @Query() dto: CommonPaginationDto,
  ) {
    return this.coinHistoryService.findAll(accountId, dto);
  }

  @Get(':id')
  findOne(@Param('id') id: string) {
    return this.coinHistoryService.findOne(+id);
  }

  @Patch(':id')
  update(
    @Param('id') id: string,
    @Body() updateCoinHistoryDto: UpdateCoinHistoryDto,
  ) {
    return this.coinHistoryService.update(+id, updateCoinHistoryDto);
  }

  @Delete(':id')
  remove(@Param('id') id: string) {
    return this.coinHistoryService.remove(+id);
  }
}
