import {
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import {
  CategoryPaginationDto,
  CategoryStatusDto,
  CreateCategoryDto,
} from './dto/create-category.dto';
import {
  CategorySetTimeDto,
  UpdateCategoryDto,
} from './dto/update-category.dto';
import { Category } from './entities/category.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Repository } from 'typeorm';
import {
  CategoryPublishStatus,
  CategoryStatus,
  DefaultStatus,
  GameType,
} from 'src/enum';
import { unlink } from 'fs/promises';
import { join } from 'path';
import { Account } from 'src/account/entities/account.entity';
import { Game } from 'src/game/entities/game.entity';

@Injectable()
export class CategoryService {
  constructor(
    @InjectRepository(Category) private readonly repo: Repository<Category>,
    @InjectRepository(Account) private readonly accRepo: Repository<Account>,
    @InjectRepository(Game) private readonly gameRepo: Repository<Game>,
  ) {}

  async create(dto: CreateCategoryDto) {
    const obj = Object.assign(dto);
    return this.repo.save(obj);
  }

  async findAll(dto: CategoryPaginationDto) {
    const keyword = dto.keyword || '';
    const query = await this.repo.createQueryBuilder('category');
    if (dto.status && dto.status.length > 0) {
      query.where('category.status = :status', { status: dto.status });
    }
    if (dto.pubStatus && dto.pubStatus.length > 0) {
      query.where('category.pubStatus = :pubStatus', {
        pubStatus: dto.pubStatus,
      });
    }
    if (dto.keyword && dto.keyword.length > 0) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where('category.name LIKE :keyword', {
            keyword: '%' + keyword + '%',
          });
        }),
      );
    }
    const [result, total] = await query
      .orderBy({ 'category.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async findByUser(accountId: string) {
    const query = await this.repo
      .createQueryBuilder('category')
      .leftJoinAndSelect(
        'category.userCategory',
        'userCategory',
        'userCategory.accountId = :accountId',
        { accountId: accountId },
      )
      .where('category.status = :status AND category.pubStatus = :pubStatus', {
        status: CategoryStatus.ACTIVE,
        pubStatus: CategoryPublishStatus.PUBLISH,
      });
    // .where(
    //   'category.status IN (:...status) AND category.pubStatus IN (:...pubStatus)',
    //   {
    //     status: [CategoryStatus.ACTIVE, CategoryStatus.DEACTIVE],
    //     pubStatus: [
    //       CategoryPublishStatus.PUBLISH,
    //       CategoryPublishStatus.UNPUBLISH,
    //     ],
    //   },
    // );
    const [result, total] = await query.getManyAndCount();

    return { result, total };
  }

  async onlyDeactive(accountId: string) {
    const now = new Date();
    const fifteenDaysEarly = new Date();
    fifteenDaysEarly.setDate(now.getDate() - 15);    

    const query = await this.repo
      .createQueryBuilder('category')
      .leftJoinAndSelect(
        'category.userCategory',
        'userCategory',
        'userCategory.accountId = :accountId',
        { accountId },
      )
      .where('category.status = :status AND category.pubStatus = :pubStatus', {
        status: CategoryStatus.DEACTIVE,
        pubStatus: CategoryPublishStatus.UNPUBLISH,
      })
      .andWhere('category.publishDateTo BETWEEN :start AND :end', {
        start: fifteenDaysEarly.toISOString().split('T')[0],
        end: now.toISOString().split('T')[0],
      });

    const [result, total] = await query.getManyAndCount();
    return { result, total };
  }

  async onlyUpcoming() {
    const query = await this.repo
      .createQueryBuilder('category')
      // .leftJoinAndSelect('category.userCategory', 'userCategory')
      .where('category.status = :status AND category.pubStatus = :pubStatus', {
        status: CategoryStatus.UPCOMING,
        pubStatus: CategoryPublishStatus.UNPUBLISH,
      });
    const [result, total] = await query.getManyAndCount();

    return { result, total };
  }

  async categoryDetail(id: string) {
    const result = await this.repo
      .createQueryBuilder('category')
      .leftJoinAndSelect('category.categoryEvidence', 'categoryEvidence')
      .leftJoinAndSelect('category.categorySuspect', 'categorySuspect')
      .where('category.id = :id', { id: id })
      .getOne();
    if (!result) {
      throw new NotFoundException('Category Not Found..');
    }
    return result;
  }

  async findOne(id: string) {
    const result = await this.repo.findOne({ where: { id: id } });
    if (!result) {
      throw new NotFoundException('Category Not Found..');
    }
    return result;
  }

  async update(id: string, dto: UpdateCategoryDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Category Not Found..');
    }
    const obj = Object.assign(result, dto);
    return await this.repo.save(obj);
  }

  async setTimer(id: string, dto: CategorySetTimeDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Category Not Found..');
    }
    const obj = Object.assign(result, {
      publishDateFrom: dto.publishDateFrom,
      catOpenTimer: dto.catOpenTimer,
      publishDateTo: dto.publishDateTo,
      catCloseTime: dto.catCloseTime,
      status: CategoryStatus.UPCOMING,
    });
    const category = await this.repo.save(obj);

    // add timer to c3 game
    const findQuickFire = await this.gameRepo.findOne({
      where: { categoryId: id, type: GameType.C3 },
    });
    if (!findQuickFire) {
      throw new NotFoundException('Quick Fire Game Not Found..');
    }
    const { publishDateFrom, gameOpenTimer, gameCloseTimeString } =
      await this.calculateGameDateTime(
        dto.publishDateFrom,
        dto.catOpenTimer,
        dto.publishDateTo,
        dto.catCloseTime,
      );
    const gameObj = Object.assign(findQuickFire, {
      publishDateFrom: publishDateFrom,
      gameOpenTimer: gameOpenTimer,
      publishDateTo: publishDateFrom,
      gameCloseTime: gameCloseTimeString,
    });
    this.gameRepo.save(gameObj);

    return category;
  }

  async icon(image: string, result: Category) {
    if (result.iconPath) {
      const oldPath = join(__dirname, '..', '..', result.iconPath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      icon: process.env.CLU_CDN_LINK + image,
      iconPath: image,
    });
    return this.repo.save(obj);
  }

  async image(image: string, result: Category) {
    if (result.imagePath) {
      const oldPath = join(__dirname, '..', '..', result.imagePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      image: process.env.CLU_CDN_LINK + image,
      imagePath: image,
    });
    return this.repo.save(obj);
  }

  async status(id: string, dto: CategoryStatusDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Category Not Found..');
    }
    if (
      result.status !== CategoryStatus.UPCOMING &&
      dto.pubStatus == CategoryPublishStatus.PUBLISH
    ) {
      throw new ConflictException('Category need to be in UPCOMING first..');
    }
    const obj = Object.assign(result, dto);
    return await this.repo.save(obj);
  }

  async remove(id: string) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Category Not Found..');
    }
    if (result.iconPath) {
      const oldPath = join(__dirname, '..', '..', result.iconPath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    return this.repo.remove(result);
  }

  async calculateGameDateTime(startDate, startTime, endDate, endTime) {
    const start = new Date(`${startDate}T${startTime}`);
    const end = new Date(`${endDate}T${endTime}`);

    /*This works for local only***/
    //Adding 5:30 Hours to follow ISTtime zone
    // const offsetMs = (5 * 60 + 30) * 60 * 1000;
    // const startWithOffset = new Date(start.getTime() + offsetMs);
    // const endWithOffset = new Date(end.getTime() + offsetMs);
    // const duration = endWithOffset.getTime() - startWithOffset.getTime();
    // const seventyFivePercent = duration * 0.75;
    // const targetTime = new Date(startWithOffset.getTime() + seventyFivePercent);

    /*This works for server only***/
    const offsetMs = 60 * 60 * 1000;
    const startWithOffset = new Date(start.getTime() + offsetMs);
    const endWithOffset = new Date(end.getTime() + offsetMs);
    const duration = endWithOffset.getTime() - startWithOffset.getTime();
    const seventyFivePercent = duration * 0.75;
    const targetTime = new Date(startWithOffset.getTime() + seventyFivePercent);

    const publishDateFrom = targetTime.toISOString().split('T')[0];
    const gameOpenTimer = targetTime.toISOString().split('T')[1].split('.')[0];
    const gameCloseTime = new Date(targetTime.getTime() + 30 * 60 * 1000);
    const gameCloseTimeString = gameCloseTime
      .toISOString()
      .split('T')[1]
      .split('.')[0];

    return {
      publishDateFrom,
      gameOpenTimer,
      gameCloseTimeString,
    };
  }
}
