import { Injectable, Logger } from '@nestjs/common';
import { Cron, CronExpression } from '@nestjs/schedule';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { Category } from './entities/category.entity';
import { CategoryStatus, CategoryPublishStatus } from 'src/enum';

@Injectable()
export class CategoryCronService {
  private readonly logger = new Logger(CategoryCronService.name);

  constructor(
    @InjectRepository(Category)
    private readonly categoryRepo: Repository<Category>,
  ) {}

  /************For Local************/
  // Run every minute
  // @Cron(CronExpression.EVERY_MINUTE)
  // async handleCategoryPublish() {
  //   const now = new Date();
  //   const currentDate = now.toISOString().split('T')[0]; // YYYY-MM-DD
  //   const currentTime = now.toTimeString().slice(0, 5); // HH:mm

  //   // Find categories to PUBLISH (time window reached)
  //   const toPublish = await this.categoryRepo
  //     .createQueryBuilder('category')
  //     .where('category.publishDateFrom = :date', { date: currentDate })
  //     .andWhere('category.catOpenTimer <= :time', { time: currentTime })
  //     .andWhere('category.status = :status', { status: CategoryStatus.UPCOMING })
  //     .andWhere('category.pubStatus = :pubStatus', {
  //       pubStatus: CategoryPublishStatus.UNPUBLISH,
  //     })
  //     .getMany();

  //   if (toPublish.length == 0) {
  //     this.logger.log(`No Categories found to Publish!`);
  //   }
  //   for (const cat of toPublish) {
  //     cat.status = CategoryStatus.ACTIVE;
  //     cat.pubStatus = CategoryPublishStatus.PUBLISH;
  //     await this.categoryRepo.save(cat);
  //     this.logger.log(`Published: ${cat.name}`);
  //   }

  //   // Find categories to UNPUBLISH (close time reached)
  //   const toUnpublish = await this.categoryRepo
  //     .createQueryBuilder('category')
  //     .where('category.publishDateTo = :date', { date: currentDate })
  //     .andWhere('category.catCloseTime <= :time', { time: currentTime })
  //     .andWhere('category.status = :status', { status: CategoryStatus.ACTIVE })
  //     .andWhere('category.pubStatus = :pubStatus', {
  //       pubStatus: CategoryPublishStatus.PUBLISH,
  //     })
  //     .getMany();

  //   if (toUnpublish.length == 0) {
  //     this.logger.log(`No Categories found to Unpublish!`);
  //   }
  //   for (const cat of toUnpublish) {
  //     cat.status = CategoryStatus.DEACTIVE;
  //     cat.pubStatus = CategoryPublishStatus.UNPUBLISH;
  //     await this.categoryRepo.save(cat);
  //     this.logger.log(`Unpublished: ${cat.name}`);
  //   }
  // }

  /************For Server************/
  @Cron(CronExpression.EVERY_MINUTE)
  async handleCategoryPublish() {
    const now = new Date();
    const later = new Date(now.getTime() + (4 * 60 + 30) * 60000); // 4h 30m in ms

    const currentDate = later.toISOString().split('T')[0];
    const currentTime = later.toTimeString().slice(0, 5);

    // Find categories to PUBLISH (time window reached)
    const toPublish = await this.categoryRepo
      .createQueryBuilder('category')
      .where('category.publishDateFrom = :date', { date: currentDate })
      .andWhere('category.catOpenTimer <= :time', { time: currentTime })
      .andWhere('category.status = :status', {
        status: CategoryStatus.UPCOMING,
      })
      .andWhere('category.pubStatus = :pubStatus', {
        pubStatus: CategoryPublishStatus.UNPUBLISH,
      })
      .getMany();

    if (toPublish.length == 0) {
      this.logger.log(`No Categories found to Publish!`);
    }
    for (const cat of toPublish) {
      cat.status = CategoryStatus.ACTIVE;
      cat.pubStatus = CategoryPublishStatus.PUBLISH;
      await this.categoryRepo.save(cat);
      this.logger.log(`Published: ${cat.name}`);
    }

    // Find categories to UNPUBLISH (close time reached)
    const toUnpublish = await this.categoryRepo
      .createQueryBuilder('category')
      .where('category.publishDateTo = :date', { date: currentDate })
      .andWhere('category.catCloseTime <= :time', { time: currentTime })
      .andWhere('category.status = :status', { status: CategoryStatus.ACTIVE })
      .andWhere('category.pubStatus = :pubStatus', {
        pubStatus: CategoryPublishStatus.PUBLISH,
      })
      .getMany();

    if (toUnpublish.length == 0) {
      this.logger.log(`No Categories found to Unpublish!`);
    }
    for (const cat of toUnpublish) {
      cat.status = CategoryStatus.DEACTIVE;
      cat.pubStatus = CategoryPublishStatus.UNPUBLISH;
      await this.categoryRepo.save(cat);
      this.logger.log(`Unpublished: ${cat.name}`);
    }
  }
}
