import { ApiProperty } from '@nestjs/swagger';
import { Type } from 'class-transformer';
import {
  IsDate,
  IsEmail,
  IsEnum,
  IsNotEmpty,
  IsNumber,
  IsOptional,
  IsString,
  Max,
  MaxLength,
  Min,
  MinLength,
} from 'class-validator';
import {
  DefaultStatus,
  Gender,
  StaffRole,
  SubscriptionStatus,
  UserRole,
  YNStatus,
} from 'src/enum';

export class CreateAccountDto {
  @ApiProperty()
  @IsNotEmpty()
  @MinLength(1)
  @MaxLength(55)
  email: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsString()
  @MinLength(5)
  @MaxLength(30)
  password: string;

  @ApiProperty()
  @IsNotEmpty()
  salutation: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsEnum(Gender)
  gender: Gender;

  @ApiProperty()
  @IsNotEmpty()
  @IsString()
  @MinLength(2)
  @MaxLength(30)
  fName: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsString()
  @MinLength(2)
  @MaxLength(30)
  lName: string;

  @ApiProperty()
  @IsNotEmpty()
  @MinLength(10)
  @MaxLength(10)
  mobile: string;

  @ApiProperty()
  @IsNotEmpty()
  @IsEnum(StaffRole)
  staffRole: StaffRole;

  @ApiProperty()
  @IsNotEmpty()
  @IsEnum(DefaultStatus)
  status: DefaultStatus;
}

export class UpdateStaffPasswordDto {
  @IsOptional()
  @IsString()
  loginId: string;

  @IsOptional()
  password: string;
}

export class UpdateStaffDto {
  @ApiProperty()
  @IsOptional()
  salutation: string;

  @ApiProperty()
  @IsOptional()
  gender: Gender;

  @ApiProperty()
  @IsOptional()
  @IsString()
  fName: string;

  @ApiProperty()
  @IsOptional()
  @IsString()
  lName: string;

  @ApiProperty()
  @IsOptional()
  mobile: string;

  @ApiProperty()
  @IsOptional()
  staffRole: StaffRole;
}

export class PaginationDto {
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(10)
  @Max(50)
  limit: number;

  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  @Max(1000)
  offset: number;

  @IsOptional()
  @MinLength(0)
  @MaxLength(100)
  keyword: string;

  @IsOptional()
  @IsEnum(DefaultStatus)
  status: DefaultStatus;
}

export class StatusDto {
  @IsNotEmpty()
  @IsEnum(DefaultStatus)
  status: DefaultStatus;
}

export class MemberPaginationDto {
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(10)
  limit: number;

  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  offset: number;

  @IsOptional()
  @MinLength(0)
  @MaxLength(100)
  keyword: string;

  @IsNotEmpty()
  status: DefaultStatus;
}

export class AccDeleteReasonDto {
  @IsNotEmpty()
  reason: string;
}

export class LeaderPaginationDto {
  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(10)
  @Max(50)
  limit: number;

  @IsNotEmpty()
  @Type(() => Number)
  @IsNumber()
  @Min(0)
  @Max(1000)
  offset: number;

  @IsOptional()
  @MinLength(0)
  @MaxLength(100)
  keyword: string;

  @IsOptional()
  month: string;

  @IsOptional()
  @Type(() => Number)
  year: number;
}
