import {
  BadRequestException,
  ConflictException,
  Inject,
  Injectable,
  NotAcceptableException,
  NotFoundException,
  UnauthorizedException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import {
  DefaultStatus,
  PlanType,
  PlayerBadge,
  PlayGameStatus,
  UserRole,
} from 'src/enum';
import { Brackets, Repository } from 'typeorm';
import {
  AccDeleteReasonDto,
  CreateAccountDto,
  LeaderPaginationDto,
  MemberPaginationDto,
  UpdateStaffDto,
  UpdateStaffPasswordDto,
} from './dto/account.dto';
import { Account } from './entities/account.entity';
import { CACHE_MANAGER } from '@nestjs/cache-manager';
import { Cache } from 'cache-manager';
import { UserDetail } from 'src/user-details/entities/user-detail.entity';
import { DefaultStatusDto } from 'src/common/dto/default-status.dto';
import * as bcrypt from 'bcrypt';
import { StaffDetail } from 'src/staff-details/entities/staff-detail.entity';
import { DefaultStatusPaginationDto } from 'src/common/dto/default-status-pagination.dto';
import { Menu } from 'src/menus/entities/menu.entity';
import { MenusService } from 'src/menus/menus.service';
import { PermissionsService } from 'src/permissions/permissions.service';
import { UserPermissionsService } from 'src/user-permissions/user-permissions.service';
import { NodeMailerService } from 'src/node-mailer/node-mailer.service';
import { Plan } from 'src/plan/entities/plan.entity';
import { PlanHistory } from 'src/plan-history/entities/plan-history.entity';
import { LeaderboardMonthly } from 'src/leaderboard-monthly/entities/leaderboard-monthly.entity';
import { CommonPaginationDto } from 'src/common/dto/common-pagination.dto';
import { UserCategory } from 'src/user-category/entities/user-category.entity';

@Injectable()
export class AccountService {
  constructor(
    @InjectRepository(Account) private readonly repo: Repository<Account>,
    @InjectRepository(UserDetail)
    private readonly udRepo: Repository<UserDetail>,
    @InjectRepository(Plan)
    private readonly planRepo: Repository<Plan>,
    @InjectRepository(PlanHistory)
    private readonly planHistRepo: Repository<PlanHistory>,
    @InjectRepository(LeaderboardMonthly)
    private readonly lbMonthRepo: Repository<LeaderboardMonthly>,
    @InjectRepository(UserCategory)
    private readonly uCatRepo: Repository<UserCategory>,
    @Inject(CACHE_MANAGER) private cacheManager: Cache,
    @InjectRepository(StaffDetail)
    private readonly staffRepo: Repository<StaffDetail>,
    @InjectRepository(Menu)
    private readonly menuRepo: Repository<Menu>,
    private readonly menuService: MenusService,
    private readonly permissionService: PermissionsService,
    private readonly userPermService: UserPermissionsService,
    private readonly nodemailerService: NodeMailerService,
  ) {}

  async create(dto: CreateAccountDto, businessAdminId: string) {
    const user = await this.repo.findOne({
      where: { email: dto.email, roles: UserRole.STAFF },
    });
    if (user) {
      throw new ConflictException('Email already exists!');
    }

    const encryptedPassword = await bcrypt.hash(dto.password, 13);
    const obj = Object.assign({
      email: dto.email,
      password: encryptedPassword,
      businessAdminId: businessAdminId,
      roles: UserRole.STAFF,
      status: dto.status,
    });
    const payload = await this.repo.save(obj);
    const object = Object.assign({
      staffId: '#' + Math.floor(1000 + Math.random() * 9000),
      salutation: dto.salutation,
      gender: dto.gender,
      fName: dto.fName,
      lName: dto.lName,
      mobile: dto.mobile,
      staffRole: dto.staffRole,
      accountId: payload.id,
    });
    await this.staffRepo.save(object);
    return payload;
  }

  async userList(dto: MemberPaginationDto) {
    const keyword = dto.keyword || '';
    const query = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.city', 'city')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.status',
        'account.totalCoin',
        'account.totalPoint',
        'account.totalXP',
        'account.badge',
        'account.referralCode',
        'account.referredBy',
        'account.createdAt',

        'userDetail.id',
        'userDetail.email',
        'userDetail.name',
        'userDetail.userName',
        'userDetail.dob',
        'userDetail.gender',
        'userDetail.avatar',

        'city.id',
        'city.name',
        'city.image',
      ])
      .where('account.roles = :roles AND account.status = :status', {
        roles: UserRole.USER,
        status: dto.status,
      });
    if (dto.keyword && dto.keyword.length > 0) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where(
            'account.phoneNumber LIKE :keyword OR userDetail.name LIKE :keyword OR userDetail.gender LIKE :keyword OR userDetail.userName LIKE :keyword ',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      );
    }
    const [result, total] = await query
      .orderBy({ 'account.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async getStaffDetails(
    dto: DefaultStatusPaginationDto,
    businessAdminId: string,
  ) {
    const keyword = dto.keyword || '';
    const query = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail')
      .select([
        'account.id',
        'account.businessAdminId',
        'account.email',
        'account.password',
        'account.roles',
        'account.status',
        'account.createdAt',

        'staffDetail.id',
        'staffDetail.staffId',
        'staffDetail.salutation',
        'staffDetail.gender',
        'staffDetail.fName',
        'staffDetail.lName',
        'staffDetail.mobile',
        'staffDetail.staffRole',
        'staffDetail.createdAt',
      ])
      .where(
        'account.roles = :roles AND account.status = :status AND account.businessAdminId = :businessAdminId',
        {
          roles: UserRole.STAFF,
          status: dto.status,
          businessAdminId: businessAdminId,
        },
      );

    const [result, total] = await query
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'account.phoneNumber LIKE :keyword OR staffDetail.fName LIKE :keyword OR staffDetail.lName LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      )
      .orderBy({ 'account.createdAt': 'DESC' })
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();

    return { result, total };
  }

  async getStaffProfile(accountId: string) {
    const staff = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail')
      .select([
        'account.id',
        'account.email',
        'account.businessAdminId',
        'account.roles',
        'account.status',

        'staffDetail.id',
        'staffDetail.staffId',
        'staffDetail.salutation',
        'staffDetail.gender',
        'staffDetail.fName',
        'staffDetail.lName',
        'staffDetail.mobile',
        'staffDetail.staffRole',
        'staffDetail.createdAt',
      ])
      .where('account.id = :id', { id: accountId })
      .getOne();

    const perms = await this.menuRepo
      .createQueryBuilder('menu')
      .leftJoinAndSelect('menu.userPermission', 'userPermission')
      .leftJoinAndSelect('userPermission.permission', 'permission')
      .where('userPermission.accountId = :accountId', {
        accountId: accountId,
      })
      .orderBy({ 'menu.title': 'ASC', 'permission.id': 'ASC' })
      .getMany();

    return { staff: staff, perms };
  }

  async userProfile(accountId: string) {
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.plan', 'plan')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.city', 'city')
      .select([
        'account.id',
        'account.planId',
        'account.subscriptionStartDate',
        'account.subscriptionEndDate',
        'account.planId',
        'account.phoneNumber',
        'account.totalCoin',
        'account.totalPoint',
        'account.totalXP',
        'account.badge',
        'account.roles',
        'account.status',
        'account.reason',
        'account.referralCode',
        'account.referredBy',
        'account.createdAt',

        'plan.id',
        'plan.planType',
        'plan.planName',
        'plan.price',
        'plan.dailyPuzzleCount',
        'plan.caseFileAccess',
        'plan.freeDailyCoin',
        'plan.showAds',
        // 'plan.desc',
        'plan.status',

        'userDetail.id',
        'userDetail.email',
        'userDetail.name',
        'userDetail.userName',
        'userDetail.dob',
        'userDetail.gender',
        'userDetail.avatar',

        'city.id',
        'city.name',
        'city.image',
      ])
      .where('account.id = :id', { id: accountId })
      .getOne();
    if (!result) {
      throw new NotFoundException('User not found!');
    }
    return result;
  }

  async getGlobalLeader() {
    const [result, total] = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .select([
        'account.id',
        'account.totalPoint',
        'account.totalXP',
        'account.badge',
        'account.createdAt',

        'userDetail.id',
        'userDetail.name',
        'userDetail.userName',
        'userDetail.dob',
        'userDetail.gender',
        'userDetail.avatar',
      ])
      .where('account.roles = :roles AND account.status = :status', {
        roles: UserRole.USER,
        status: DefaultStatus.ACTIVE,
      })
      .orderBy({ 'account.totalXP': 'DESC' })
      .getManyAndCount();

    return { result, total };
  }

  async getMonthlyLeader(dto: LeaderPaginationDto) {
    const date = new Date();
    const currMonth = date
      .toLocaleString('en-US', { month: 'long' })
      .toUpperCase();
    const currYear = date.getFullYear();

    const keyword = dto.keyword || '';
    const query = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('account.leaderboardMonthly', 'leaderboardMonthly')
      .select([
        'account.id',
        'account.badge',
        'account.createdAt',

        'userDetail.id',
        'userDetail.name',
        'userDetail.userName',
        'userDetail.dob',
        'userDetail.gender',
        'userDetail.avatar',

        'leaderboardMonthly.id',
        'leaderboardMonthly.monthlyTotalPoint',
        'leaderboardMonthly.monthlyXP',
        'leaderboardMonthly.month',
        'leaderboardMonthly.year',
      ])
      .where('account.roles = :roles AND account.status = :status', {
        roles: UserRole.USER,
        status: DefaultStatus.ACTIVE,
      });
    if (dto.month && dto.year && dto.month.length > 0) {
      query.andWhere(
        'leaderboardMonthly.month = :month AND leaderboardMonthly.year = :year',
        {
          month: dto.month,
          year: dto.year,
        },
      );
    } else {
      query.andWhere(
        'leaderboardMonthly.month = :month AND leaderboardMonthly.year = :year',
        {
          month: currMonth,
          year: currYear,
        },
      );
    }
    const [result, total] = await query
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'userDetail.name LIKE :keyword OR userDetail.userName LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      )
      .orderBy({ 'leaderboardMonthly.monthlyXP': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async getLeaderDetail(accountId: string) {
    const [userCategory, total] = await this.uCatRepo.findAndCount({
      where: { accountId },
    });
    var totalAccuracy = 0;
    var totalDuration = 0;
    userCategory.forEach((item) => {
      totalAccuracy += item.accuracy;
      totalDuration += item.avgDuration;
    });
    const accuracy = totalAccuracy / total;
    const avgDuration = totalDuration / total;

    const userCategorySolvedCount = await this.uCatRepo.count({
      where: { accountId, status: PlayGameStatus.COMPLETED },
    });

    const result = await this.repo
      .createQueryBuilder('account')
      .where('account.id = :id', { id: accountId })
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .select([
        'account.id',
        'account.totalXP',
        'account.badge',
        'account.winStreak',
        'account.createdAt',

        'userDetail.id',
        'userDetail.name',
        'userDetail.userName',
        'userDetail.dob',
        'userDetail.gender',
        'userDetail.avatar',
      ])
      .getOne();

    return {
      ...result,
      accuracy,
      casesSolved: userCategorySolvedCount,
      avgTime: avgDuration,
    };
  }

  async updateStaff(accountId: string, dto: UpdateStaffDto) {
    const result = await this.staffRepo.findOne({ where: { accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found With This ID.');
    }
    const obj = Object.assign(result, dto);
    return this.staffRepo.save(obj);
  }

  async updateStaffPassword(accountId: string, dto: UpdateStaffPasswordDto) {
    const result = await this.repo.findOne({ where: { id: accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found With This ID.');
    }
    if (dto.loginId && dto.loginId.length > 0) {
      const obj = Object.assign(result, { phoneNumber: dto.loginId });
      await this.repo.save(obj);
    }
    if (dto.password && dto.password.length > 0) {
      const password = await bcrypt.hash(dto.password, 10);
      const obj = Object.assign(result, { password: password });
      await this.repo.save(obj);
    }
    return result;
  }

  async status(id: string, dto: DefaultStatusDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Account Not Found With This ID.');
    }
    if (dto.status == DefaultStatus.ACTIVE) {
      result.reason = null;
      const obj = Object.assign(result, dto);
      return this.repo.save(obj);
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async statusByUser(id: string, dto: AccDeleteReasonDto) {
    try {
      const result = await this.repo.findOne({ where: { id } });
      if (!result) {
        throw new NotFoundException('Feedback not found!');
      }
      const obj = Object.assign(result, {
        reason: dto.reason,
        status: DefaultStatus.DEACTIVE,
      });
      return this.repo.save(obj);
    } catch (error) {
      console.error(error);
      throw new NotAcceptableException('Something went wrong! Try again!');
    }
  }

  async buyPlan(planId: string, accountId: string) {
    const result = await this.repo.findOne({ where: { id: accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found!');
    }
    const plan = await this.planRepo.findOne({ where: { id: planId } });
    if (!plan) {
      throw new NotFoundException('Plan Not Found!');
    }
    const today = new Date();
    const duration = plan.planType == PlanType.YEARLY ? 365 : 30;
    const startDate = new Date();
    const endDate = new Date(today);
    endDate.setDate(today.getDate() + duration - 1);

    const obj = Object.assign(result, {
      planId: planId,
      subscriptionStartDate: startDate,
      subscriptionEndDate: endDate,
      totalCoin: result.totalCoin + plan.freeDailyCoin,
    });
    const phObj = Object.assign({
      planId: planId,
      accountId: accountId,
      purchaseDate: startDate,
    });
    this.planHistRepo.save(phObj);

    return this.repo.save(obj);
  }

  async addCoin(coin: number, accountId: string) {
    const result = await this.repo.findOne({ where: { id: accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found!');
    }
    const obj = Object.assign(result, { totalCoin: result.totalCoin + coin });
    return this.repo.save(obj);
  }

  async deductCoin(coin: number, accountId: string) {
    const result = await this.repo.findOne({ where: { id: accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found!');
    }
    if (result.totalCoin < coin) {
      throw new BadRequestException('Insufficient Coins!');
    }
    const obj = Object.assign(result, { totalCoin: result.totalCoin - coin });
    return this.repo.save(obj);
  }

  async addPoint(point: number, accountId: string) {
    const date = new Date();
    const currMonth = date
      .toLocaleString('en-US', { month: 'long' })
      .toUpperCase();
    const currYear = date.getFullYear();

    const result = await this.repo.findOne({ where: { id: accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found!');
    }
    const addedXP = Math.ceil(point / 200);

    const totalXP = result.totalXP + addedXP;
    let newBadge = result.badge;
    if (result.badge !== PlayerBadge.Founding_Member) {
      if (totalXP < 500) newBadge = PlayerBadge.Rookie;
      else if (totalXP < 1500) newBadge = PlayerBadge.Skilled;
      else if (totalXP < 3000) newBadge = PlayerBadge.Pro;
      else if (totalXP < 6000) newBadge = PlayerBadge.Elite;
      else newBadge = PlayerBadge.Legend;
    }
    const obj = Object.assign(result, {
      totalPoint: result.totalPoint + point,
      totalXP: totalXP,
      badge: newBadge,
    });
    const account = await this.repo.save(obj);

    const check = await this.lbMonthRepo.findOne({
      where: { accountId, month: currMonth, year: currYear },
    });
    if (!check) {
      const lbObj = Object.assign({
        accountId: accountId,
        monthlyTotalPoint: point,
        monthlyXP: addedXP,
        month: currMonth,
        year: currYear,
      });
      this.lbMonthRepo.save(lbObj);
    } else {
      const monthlyXP = check.monthlyXP + addedXP;
      const lbObj = Object.assign(check, {
        monthlyTotalPoint: check.monthlyTotalPoint + point,
        monthlyXP: monthlyXP,
      });
      this.lbMonthRepo.save(lbObj);
    }

    return account;
  }

  async monthlyReset() {
    const accounts = await this.repo.find({ where: { roles: UserRole.USER } });
    accounts.forEach(async (account) => {
      const obj = Object.assign(account, {
        monthlyTotalPoint: 0,
        monthlyXP: 0,
      });
      await this.repo.save(obj);
    });
    return {
      success: true,
      message: 'Monthly Points & XPs Reset Successfully!',
    };
  }

  async addDailyCoin(accountId: string) {
    const account = await this.repo.findOne({ where: { id: accountId } });
    if (!account) {
      throw new NotFoundException('Account Not Found!');
    }
    const today = new Date().toISOString().split('T')[0];
    const lastDate = account.lastFreeCoinDate ? account.lastFreeCoinDate : null;

    if (!account.planId) {
      if (lastDate !== today) {
        account.totalCoin += 20;
        account.lastFreeCoinDate = new Date().toISOString().split('T')[0];
        await this.repo.save(account);
        return { added: 20, totalCoin: account.totalCoin };
      }
      return { added: 0, totalCoin: account.totalCoin };
    } else {
      const plan = await this.planRepo.findOne({
        where: { id: account.planId },
      });
      if (!plan) {
        throw new NotFoundException('Plan Not Found!');
      }
      if (lastDate !== today) {
        account.totalCoin += plan.freeDailyCoin;
        account.lastFreeCoinDate = new Date().toISOString().split('T')[0];
        await this.repo.save(account);
        return { added: plan.freeDailyCoin, totalCoin: account.totalCoin };
      }
      return { added: 0, totalCoin: account.totalCoin };
    }
  }

  async increaseWinStreak(accountId: string) {
    const account = await this.repo.findOne({ where: { id: accountId } });
    if (!account) {
      throw new NotFoundException('Account Not Found!');
    }
    account.winStreak += 1;
    await this.repo.save(account);
    return account;
  }

  async resetWinStreak(accountId: string) {
    const account = await this.repo.findOne({ where: { id: accountId } });
    if (!account) {
      throw new NotFoundException('Account Not Found!');
    }
    account.winStreak = 0;
    await this.repo.save(account);
    return account;
  }
}
