import {
  ConflictException,
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { TicketStatus } from 'src/enum';
import { Brackets, Repository } from 'typeorm';
import { PaginationAllDto, TicketAnswerDto, TicketDto, PaginationReportDto } from './dto/ticket.dto';
import { Ticket } from './entities/ticket.entity';

@Injectable()
export class TicketsService {
  constructor(
    @InjectRepository(Ticket) private readonly repo: Repository<Ticket>,
  ) {}

  async create(dto: TicketDto) {
    const result = await this.repo.count({
      where: {
        accountId: dto.accountId,
        status: TicketStatus.PENDING,
        createdAt: new Date(),
      },
    });
    if (result > 9) {
      throw new ConflictException('Only 10 ticket allowed at single day!');
    }
    const obj = Object.create(dto);
    return this.repo.save(obj);
  }

  async findAll(dto: PaginationAllDto) {
    const keyword = dto.keyword || '';
    const [result, total] = await this.repo
      .createQueryBuilder('ticket')
      .where('ticket.status = :status', { status: dto.status })
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'ticket.question LIKE :question OR ticket.answer LIKE :answer',
            {
              question: '%' + keyword + '%',
              answer: '%' + keyword + '%',
            },
          );
        }),
      )
      .orderBy({ 'ticket.createdAt': 'DESC' })
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();

    return { result, total };
  }

  // this is for report section
  async findReportAll(dto: PaginationReportDto) {
    const fromDate = new Date(dto.fromDate);
    fromDate.setHours(0, 0, 0, 0);

    const toDate = new Date(dto.toDate);
    toDate.setHours(23, 59, 59, 59);

    const query = this.repo
      .createQueryBuilder('ticket')
      .leftJoinAndSelect('ticket.account', 'account')
      .leftJoinAndSelect('account.outletDetail', 'outletDetail')
      .select([
        'account.id',
        'account.loginId',

        'outletDetail.id',
        'outletDetail.storeName',

        'ticket.id',
        'ticket.image',
        'ticket.question',
        'ticket.answer',
        'ticket.status',
        'ticket.createdAt'
      ]);
      query.where(
        'ticket.createdAt >= :fromDate and ticket.createdAt <= :toDate',
        { fromDate: fromDate, toDate: toDate },
      );
    const [result, total] = await query
      .orderBy({ 'ticket.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();
    return { result, total };
  }

  async find(id: string, dto: PaginationAllDto) {
    const keyword = dto.keyword || '';
    const [result, total] = await this.repo
      .createQueryBuilder('ticket')
      .where('ticket.status = :status AND ticket.accountId = :accountId', {
        status: dto.status,
        accountId: id,
      })
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'ticket.question LIKE :question OR ticket.answer LIKE :answer',
            {
              question: '%' + keyword + '%',
              answer: '%' + keyword + '%',
            },
          );
        }),
      )
      .orderBy({ 'ticket.createdAt': 'DESC' })
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();

    return { result, total };
  }

  async findOne(id: string) {
    const result = await this.repo.count({ where: { id } });
    if (!result) {
      throw new NotFoundException('Ticket not found!');
    }
    return result;
  }

  async update(id: string, dto: TicketAnswerDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Ticket not found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

}
