import {
  Body,
  Controller,
  Get,
  Param,
  Patch,
  Post,
  Put,
  Query,
  UseGuards,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { PaginationDto } from 'src/account/dto/account.dto';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { BoolStatusDto } from 'src/common/dto/bool-status.dto';
import { PermissionAction, UserRole } from 'src/enum';
import { AreaService } from './area.service';
import { AreaDto, PaginationSDto } from './dto/area.dto';

@Controller('area')
export class AreaController {
  constructor(private readonly areaService: AreaService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.CREATE, 'area'])
  create(@Body() dto: AreaDto) {
    return this.areaService.create(dto);
  }

  @Get('list/all/:cityId')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'area'])
  findAll(@Query() query: PaginationSDto, @Param('cityId') cityId: string) {
    const keyword = query.keyword || '';
    return this.areaService.findAll(
      query.limit,
      query.offset,
      keyword,
      query.status,
      +cityId,
    );
  }

  @Get('list/:cityId')
  find(@Query() query: PaginationDto, @Param('cityId') cityId: string) {
    const keyword = query.keyword || '';
    return this.areaService.find(query.limit, query.offset, keyword, +cityId);
  }

  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'area'])
  update(@Param('id') id: string, @Body() dto: AreaDto) {
    return this.areaService.update(+id, dto);
  }

  @Put('status/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'area'])
  status(@Param('id') id: string, @Body() dto: BoolStatusDto) {
    return this.areaService.status(+id, dto);
  }
}
