import { CACHE_MANAGER } from '@nestjs/cache-manager';
import {
  Inject,
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Cache } from 'cache-manager';
import { Menu } from 'src/menus/entities/menu.entity';
import { Repository } from 'typeorm';
import { StaffDetailDto, UpdateStaffDetailDto } from './dto/staff-detail.dto';
import { StaffDetail } from './entities/staff-detail.entity';

@Injectable()
export class StaffDetailsService {
  constructor(
    @InjectRepository(StaffDetail)
    private readonly repo: Repository<StaffDetail>,
    @InjectRepository(Menu)
    private readonly menuRepo: Repository<Menu>,
    @Inject(CACHE_MANAGER) private readonly cacheManager: Cache,
  ) {}

  async create(dto: StaffDetailDto) {
    const user = await this.repo.findOne({
      where: { accountId: dto.accountId },
    });

    if (user) {
      try {
        if (user) {
          const obj = Object.assign(user, dto);
          return this.repo.save(obj);
        } else {
          const obj = Object.create(dto);
          return this.repo.save(obj);
        }
      } catch (error) {
        throw new NotAcceptableException(
          'Either duplicate email or invalid details!',
        );
      }
    }
  }

  async update(accountId: string, dto: UpdateStaffDetailDto) {
    const user = await this.repo.findOne({
      where: { accountId: accountId },
    });
    if (!user) {
      throw new NotFoundException('Account not found!');
    }
    try {
      this.delStaffDetail(user.accountId);
      const obj = Object.assign(user, dto);
      return this.repo.save(obj);
    } catch (error) {
      throw new NotAcceptableException(
        'Either duplicate email/pan number/aadhar number or invalid details!',
      );
    }
  }

  async findOne(accountId: string) {
    const user = await this.repo.findOne({ where: { accountId } });
    if (!user) {
      throw new NotFoundException('User not found!');
    }
    return user;
  }

  profile(id: string) {
    return this.getStaffDetail(id);
  }

  async image(accountId: string, image: string) {
    const user = await this.repo.findOne({
      where: [{ accountId }, { id: accountId }],
    });
    if (!user) {
      throw new NotFoundException('User not found!');
    }
    const obj = Object.assign(user, {
      profile: process.env.CDN_LINK + image,
      profileName: image,
    });

    this.delStaffDetail(accountId);
    return this.repo.save(obj);
  }

  async pan(accountId: string, image: string) {
    const user = await this.repo.findOne({
      where: [{ accountId }, { id: accountId }],
    });
    if (!user) {
      throw new NotFoundException('User not found!');
    }
    const obj = Object.assign(user, {
      pan: process.env.CDN_LINK + image,
      panName: image,
    });

    this.delStaffDetail(accountId);
    return this.repo.save(obj);
  }

  async aadhar(accountId: string, image: string) {
    const user = await this.repo.findOne({ where: [{ accountId }, {id: accountId}] });
    if (!user) {
      throw new NotFoundException('User not found!');
    }
    const obj = Object.assign(user, {
      aadhar: process.env.CDN_LINK + image,
      aadharName: image,
    });

    this.delStaffDetail(accountId);
    return this.repo.save(obj);
  }

  private delStaffDetail(id: string) {
    this.cacheManager.del('staffDetail' + id);
  }

  private async getStaffDetail(id: string) {
    const user = await this.repo
        .createQueryBuilder('staffDetail')
        .leftJoinAndSelect('staffDetail.account', 'account')
        .where('staffDetail.accountId = :accountId', { accountId: id })
        .getOne();
    
    if (!user) {
      throw new NotFoundException('Account details not found!');
    }
    
    const perms = await this.menuRepo
        .createQueryBuilder('menu')
        .leftJoinAndSelect('menu.userPermission', 'userPermission')
        .leftJoinAndSelect('userPermission.permission', 'permission')
        .where('userPermission.accountId = :accountId', { accountId: id })
        .orderBy({ 'menu.title': 'ASC', 'permission.id': 'ASC' })
        .getMany();

    return { user, perms };
  }
}
