import {
  Body,
  Controller,
  Get,
  Headers,
  Param,
  Patch,
  Put,
  Query,
  UseGuards,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PermissionAction, UserRole } from 'src/enum';
import { NotifyService } from 'src/notify/notify.service';
import { PaginationDto } from './dto/pagination.dto';
import { SettingDto } from './dto/setting.dto';
import { StatusSettingDto } from './dto/status-setting.dto';
import { SettingsService } from './settings.service';

@Controller('settings')
export class SettingsController {
  version = new Date();
  constructor(
    private readonly settingsService: SettingsService,
    private readonly notifyService: NotifyService,
  ) {}

  @Get('default')
  findSettings(@Headers('origin') origin: string) {
    return this.settingsService.findSetting('http://localhost:3000');
  }

  @Get()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF, UserRole.SUPER_ADMIN)
  @CheckPermissions([PermissionAction.READ, 'setting'])
  findAll(@Query() query: PaginationDto) {
    return this.settingsService.findAll(query);
  }

  @Get(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF, UserRole.SUPER_ADMIN)
  @CheckPermissions([PermissionAction.READ, 'setting'])
  findOne(@Param('id') id: string) {
    return this.settingsService.findOne(id);
  }

  @Get('appsetting/:id')
  findOneSetting(@Param('id') id: string) {
    return this.settingsService.getAppSetting(id);
  }

  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF, UserRole.SUPER_ADMIN)
  @CheckPermissions([PermissionAction.UPDATE, 'setting'])
  update(@Param('id') id: string, @Body() dto: SettingDto) {
    if (dto.maintenance === true) {
      this.notifyService.appMaintenance();
    }
    return this.settingsService.update(id, dto);
  }

  @Put(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.STAFF, UserRole.SUPER_ADMIN)
  @CheckPermissions([PermissionAction.UPDATE, 'setting'])
  status(@Param('id') id: string, @Body() dto: StatusSettingDto) {
    return this.settingsService.status(id, dto);
  }
}
