import {
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { AccountService } from 'src/account/account.service';
import { NotifyService } from 'src/notify/notify.service';
import {
  PaginationDto,
  StatusDto,
} from 'src/outlet-branch/dto/outlet-branch.dto';
import { Brackets, Repository } from 'typeorm';
import {
  CreateOutletDetailDto,
  OutletDetailDto,
  TimeIncreaseDto,
} from './dto/outlet-detail.dto';
import { OutletDetail } from './entities/outlet-detail.entity';
import { RouteAssginLog } from 'src/route-assgin-log/entities/route-assgin-log.entity';

@Injectable()
export class OutletDetailsService {
  constructor(
    @InjectRepository(OutletDetail)
    private readonly repo: Repository<OutletDetail>,
    @InjectRepository(RouteAssginLog)
    private readonly ralRepo: Repository<RouteAssginLog>,
    private readonly accountService: AccountService,
    private readonly notifyService: NotifyService,
  ) {}

  async create(dto: CreateOutletDetailDto, createdBy: string) {
    const account = await this.accountService.outletDetail(
      dto.loginId,
      dto.password,
      createdBy,
    );
    const result = await this.repo.findOne({
      where: {
        storeName: dto.storeName,
        contactPerson: dto.contactPerson,
        address: dto.address,
        latitude: dto.latitude,
        longitude: dto.longitude,
        outletBranchId: dto.outletBranchId,
      },
    });
    if (result) {
      throw new ConflictException('Outlet already exists!');
    }
    dto.accountId = account.id;
    dto.outletId = await Date.now().toString();
    const obj = Object.create(dto);
    return this.repo.save(obj);
  }

  async findAll(dto: any) {
    const keyword = dto.keyword || '';
    const [result, total] = await this.repo
      .createQueryBuilder('outletDetail')
      .leftJoinAndSelect('outletDetail.outletBranch', 'outletBranch')
      .leftJoinAndSelect('outletDetail.routeMaster', 'routeMaster')
      .leftJoinAndSelect('outletDetail.account', 'outletAccount')
      .leftJoinAndSelect('routeMaster.account', 'account')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail')
      .leftJoinAndSelect('outletDetail.area', 'area')
      .leftJoinAndSelect('outletDetail.city', 'city')
      .leftJoinAndSelect('outletDetail.state', 'state')
      .where('outletDetail.status = :status', {
        status: dto.status,
      })
      .andWhere(
        new Brackets((qb) => {
          qb.where('outletDetail.storeName LIKE :storeName', {
            storeName: '%' + keyword + '%',
          });
        }),
      )
      .skip(dto.offset)
      .take(dto.limit)
      .orderBy({ 'outletDetail.storeName': 'ASC' })
      .getManyAndCount();
    return { result, total };
  }

  async findOrderId(id: string) {
    return this.repo
      .createQueryBuilder('outletDetail')
      .leftJoinAndSelect('outletDetail.outletBranch', 'outletBranch')
      .leftJoinAndSelect('outletBranch.companyDetail', 'companyDetail')
      .leftJoinAndSelect('outletDetail.area', 'area')
      .select([
        'outletDetail.id',
        'outletDetail.storeName',
        'outletBranch.id',
        'outletBranch.branchName',
        'companyDetail.id',
        'companyDetail.businessName',
        'area.id',
        'area.name',
      ])
      .where('outletDetail.accountId = :accountId', { accountId: id })
      .getOne();
  }

  async find(dto: any, branchId: string) {
    const keyword = dto.keyword || '';
    const [result, total] = await this.repo
      .createQueryBuilder('outletDetail')
      .leftJoinAndSelect('outletDetail.outletBranch', 'outletBranch')
      .leftJoinAndSelect('outletDetail.routeMaster', 'routeMaster')
      .leftJoinAndSelect('routeMaster.account', 'account')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail')
      .leftJoinAndSelect('outletDetail.area', 'area')
      .leftJoinAndSelect('outletDetail.city', 'city')
      .leftJoinAndSelect('outletDetail.state', 'state')
      .where('outletDetail.outletBranchId = :outletBranchId AND outletDetail.status = :status', {
        outletBranchId: branchId,
        status: dto.status,
      })
      .andWhere(
        new Brackets((qb) => {
          qb.where('outletDetail.storeName LIKE :storeName', {
            storeName: '%' + keyword + '%',
          });
        }),
      )
      .skip(dto.offset)
      .take(dto.limit)
      .orderBy({ 'outletDetail.storeName': 'ASC' })
      .getManyAndCount();
    return { result, total };
  }

  async findOne(id: string) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Outlet not found!');
    }
    return result;
  }

  async outletProfile(id: string) {
    const result = await this.repo
    .createQueryBuilder('outletDetail')
    .leftJoinAndSelect('outletDetail.outletBranch', 'outletBranch')
    .leftJoinAndSelect('outletDetail.account', 'account')
    .leftJoinAndSelect('outletDetail.routeMaster', 'routeMaster')
    .leftJoinAndSelect('outletDetail.area', 'area')
    .leftJoinAndSelect('outletDetail.city', 'city')
    .leftJoinAndSelect('outletDetail.state', 'state')
      .where('outletDetail.id = :id', {
        id: id
      })
      .getOne();

    if (!result) {
      throw new NotFoundException('Profile not found!');
    }
    return result;
  }

  async update(id: string, dto: OutletDetailDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Outlet not found!');
    }
    // save log
    if(result.routeMasterId != dto.routeMasterId){
      const object = Object.create({
        fromRoute: result.routeMasterId,
        toRoute: dto.routeMasterId,
        outletDetailId: result.id,
      });
      this.ralRepo.save(object);
    }
    const obj = Object.assign(result, dto);
    const payload = await this.repo.save(obj);
    return payload;
  }

  async status(id: string, dto: StatusDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Outlet not found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async timeIncrease(id: string, dto: TimeIncreaseDto) {
     // for outlet record
     const resultoutlet = await this.repo.findOne({ where: { id } });

     // for account record
     const resultacccount = await this.repo.findOne({
       relations: ['account'],
       where: { id },
     });

     if (!resultacccount) {
       throw new NotFoundException('Outlet not found!');
     }
     this.notifyService.timeIncreaseReply(
      resultacccount.account['fcm'],
      resultacccount.account['id'],
      resultoutlet.id,
     );

    const obj = Object.assign(resultacccount, dto);
    return this.repo.save(obj);
  }
}
