import {
  Body,
  Controller,
  Delete,
  FileTypeValidator,
  Get,
  MaxFileSizeValidator,
  Param,
  ParseFilePipe,
  Patch,
  Post,
  Put,
  Query,
  UploadedFile,
  UseGuards,
  UseInterceptors,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { FileInterceptor } from '@nestjs/platform-express';
import { Account } from 'src/account/entities/account.entity';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { UserRole } from 'src/enum';
import {
  DeliveryBoyDto,
  JsonOrderDto,
  OrderDto,
  OrderJsonDto,
  PaginationAllDto,
  PaginationAllReportDto,
  PaginationReportDto,
  PaginationDto,
  RemarkDto,
  SearchListDto,
  StatusDto,
  StatusOrderDto,
} from './dto/order.dto';
import { OrdersService } from './orders.service';

@Controller('orders')
export class OrdersController {
  constructor(private readonly ordersService: OrdersService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.SUB_OUTLET,
  )
  create(@Body() dto: OrderDto, @CurrentUser() user: Account) {
    dto.outletDetailId = user.outletDetail[0].id;
    return this.ordersService.create(dto, user.outletDetail[0].id);
  }

  @Get('admin/list')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  adminOrderList(@Query() query: PaginationAllDto) {
    return this.ordersService.adminOrderList(query);
  }

  @Get('admin/report')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  adminOrderReport(@Query() query: PaginationAllReportDto) {
    return this.ordersService.adminOrderReport(query);
  }
  
  @Get('/report')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  findReportAll(@Query() dto: PaginationReportDto) {
    return this.ordersService.findReportAll(dto);
  }

  @Get('cart')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.SUB_OUTLET,
  )
  outletCartList(@CurrentUser() user: Account, @Query() dto: StatusDto) {
    return this.ordersService.outletCartList(user.outletDetail[0].id, dto);
  }

  @Get('outlet/list')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.SUB_OUTLET,
  )
  outletOrderList(@CurrentUser() user: Account, @Query() query: PaginationDto) {
    return this.ordersService.outletOrderList(user.outletDetail[0].id, query);
  }

  @Get('list/:cartId')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.SUB_OUTLET,
    UserRole.DELIVERY_BOY,
  )
  outletOrderItems(@Param('cartId') cartId: string) {
    return this.ordersService.outletOrderItems(cartId);
  }

  @Get('print/:cartId')
  printOrderItems(@Param('cartId') cartId: string) {
    return this.ordersService.printOrderItems(cartId);
  }

  @Get('delivery-boy/history')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.DELIVERY_BOY,
  )
  deliveryBoyHistory(
    @CurrentUser() user: Account,
    @Query() query: PaginationDto,
  ) {
    return this.ordersService.findGroupDeliveryHistory(user.id, query);
  }

  @Get('delivery-boy/list')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.DELIVERY_BOY,
  )
  deliveryBoyList(@CurrentUser() user: Account, @Query() dto: SearchListDto) {
    return this.ordersService.findGroupByDelivery(user.id, dto);
  }

  @Get(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.SUB_OUTLET,
  )
  findOne(@Param('id') id: string) {
    return this.ordersService.findOne(id);
  }

  @Patch()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.SUB_OUTLET,
  )
  update(@Body() dto: OrderJsonDto, @CurrentUser() user: Account) {
    let date = new Date(dto.date);
    if (!dto.status) {
      let currentDate = new Date();
      currentDate.setDate(currentDate.getDate() + 1);
      if(date === currentDate){
        date.setDate(date.getDate() + 1);
      }
    }
    return this.ordersService.update(date, dto.data, user.id, user.fcm);
  }

  @Patch('assign-delivery-boy/:cartId')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  assignDeliveryBoy(
    @Param('cartId') cartId: string,
    @Body() dto: DeliveryBoyDto,
  ) {
    return this.ordersService.updateDeliveryBoy(cartId, dto.accountId, dto.status);
  }

  @Put(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.DELIVERY_BOY,
  )
  updateQty(@Param('id') id: string, @Body() dto: StatusOrderDto) {
    return this.ordersService.updateQty(id, dto);
  }

  @Put('remark/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.SUB_OUTLET)
  remark(@Param('id') id: string, @Body() dto: RemarkDto) {
    return this.ordersService.remark(id, dto);
  }

  @Put('return-product/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.DELIVERY_BOY,
  )
  @UseInterceptors(FileInterceptor('file'))
  async image1(
    @Param('id') id: string,
    @UploadedFile(
      new ParseFilePipe({
        validators: [
          new FileTypeValidator({ fileType: '.(png|jpeg|jpg)' }),
          new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 }),
        ],
      }),
    )
    file: Express.Multer.File,
  ) {
    return this.ordersService.returnProduct(id, file.path);
  }

  @Put('signature/:cartId')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.DELIVERY_BOY,
  )
  @UseInterceptors(FileInterceptor('file'))
  async image2(
    @Param('cartId') cartId: string,
    @UploadedFile(
      new ParseFilePipe({
        validators: [
          new FileTypeValidator({ fileType: '.(png|jpeg|jpg)' }),
          new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 }),
        ],
      }),
    )
    file: Express.Multer.File,
  ) {
    return this.ordersService.signature(cartId, file.path);
  }

  @Put()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.DELIVERY_BOY,
  )
  status(@Body() dto: JsonOrderDto) {
    return this.ordersService.status(dto.data);
  }

  @Delete('/admin:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.SUB_OUTLET,
  )
  removeorders(@Param('id') id: string) {
    console.log(id)
    return this.ordersService.removeorders(id);
  }

  @Delete(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(
    UserRole.ADMIN,
    UserRole.SUPER_ADMIN,
    UserRole.STAFF,
    UserRole.SUB_OUTLET,
  )
  removesingle(@Param('id') id: string) {
    return this.ordersService.removesingle(id);
  }
}
