import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import * as schedule from 'node-schedule';
import { NotificationType } from 'src/enum';
import { Notification } from 'src/notifications/entities/notification.entity';
import { OutletDetail } from 'src/outlet-details/entities/outlet-detail.entity';
import { sendNotifications } from 'src/utils/notify.utils';
import { Repository } from 'typeorm';
import { CreateNotifyDto } from './dto/create-notify.dto';

@Injectable()
export class CronNotifyService {
  constructor(
    @InjectRepository(Notification)
    private readonly repo: Repository<Notification>,
    @InjectRepository(OutletDetail)
    private readonly orepo: Repository<OutletDetail>,
  ) {
    this.oneHourBefore();
  }

  async oneHourBefore() {
    schedule.scheduleJob(
      { hour: 21, minute: 30, tz: 'Asia/Kolkata' },
      async () => {
        try {
          const result = await this.orepo
            .createQueryBuilder('outletDetail')
            .leftJoinAndSelect('outletDetail.account', 'account')
            .select([
              'outletDetail.id',
              'outletDetail.storeName',
              'outletDetail.orderCloseTime',
              'outletDetail.orderOpenTime',
              'account.id',
              'account.fcm',
            ])
            .getMany();

          for (let index = 0; index < result.length; index++) {
            const closeTime = result[index].orderCloseTime;
            const outletName = result[index].storeName;
            const fcm = result[index].account['fcm'];
            const accountId = result[index].account['id'];
            const [openHours, openMinutes] = result[index].orderOpenTime
              .toString()
              .split(':')
              .map(Number);
            const [closeHours, closeMinutes] = result[index].orderCloseTime
              .toString()
              .split(':')
              .map(Number);

            // Create a job using the defined rule
            this.orderOpenTime60Before(
              openHours,
              openMinutes,
              fcm,
              outletName,
              accountId,
            );

            // Create a job using the defined rule
            this.orderOpenTime30Before(
              openHours,
              openMinutes,
              fcm,
              outletName,
              accountId,
            );

            // Create a job using the defined rule
            this.orderOpenTimeNow(
              openHours,
              openMinutes,
              fcm,
              outletName,
              closeTime,
              accountId,
            );

            // Create a job using the defined rule
            this.orderCloseTime60After(
              closeHours,
              closeMinutes,
              fcm,
              outletName,
              closeTime,
              accountId,
            );

            this.orderCloseTime15After(
              closeHours,
              closeMinutes,
              fcm,
              outletName,
              closeTime,
              accountId,
            );
          }
        } catch (error) {
          console.error('Error occurred in the scheduled job:', error);
        }
      },
    );
  }

  private async orderOpenTime60Before(
    hours: number,
    minutes: number,
    fcm: string,
    outletName: string,
    accountId: string,
  ) {
    const jobRule = new schedule.RecurrenceRule();
    const currentTime = new Date();

    currentTime.setHours(hours);
    currentTime.setMinutes(minutes);
    // for 60 min before + server 5.30 hrs
    // currentTime.setMinutes(currentTime.getMinutes() - 390);
    currentTime.setMinutes(currentTime.getMinutes() - 60);

    jobRule.hour = currentTime.getHours();
    jobRule.minute = currentTime.getMinutes();
    jobRule.tz = 'Asia/Kolkata'; // Set the time zone to Indian Standard Time (IST)

    schedule.scheduleJob(jobRule, () => {
      sendNotifications(
        fcm,
        'Order Open Time',
        'Hey ' +
          outletName +
          ' Your order is opening in 60 Min. Check your Order & add in bucket.',
      );
      this.saveNotification({
        title: 'Order Open Time',
        desc:
          'Hey ' +
          outletName +
          ' Your order is opening in 60 Min. Check your Order & add in bucket.',
        type: NotificationType.ORDER,
        redirectId: null,
        accountId,
      });
    });
  }

  private async orderOpenTime30Before(
    hours: number,
    minutes: number,
    fcm: string,
    outletName: string,
    accountId: string,
  ) {
    const jobRule = new schedule.RecurrenceRule();
    const currentTime = new Date();

    currentTime.setHours(hours);
    currentTime.setMinutes(minutes);
    // for 30 min before + server 5.30 hrs
    // currentTime.setMinutes(currentTime.getMinutes() - 360);
    currentTime.setMinutes(currentTime.getMinutes() - 30);
    jobRule.hour = currentTime.getHours();
    jobRule.minute = currentTime.getMinutes();
    jobRule.tz = 'Asia/Kolkata'; // Set the time zone to Indian Standard Time (IST)

    schedule.scheduleJob(jobRule, () => {
      sendNotifications(
        fcm,
        'Order Open Time',
        'Hey ' +
          outletName +
          ' Your order is opening in 30 Min. Check your Order & add in bucket.',
      );
      this.saveNotification({
        title: 'Order Open Time',
        desc:
          'Hey ' +
          outletName +
          ' Your order is opening in 30 Min. Check your Order & add in bucket.',
        type: NotificationType.ORDER,
        redirectId: null,
        accountId,
      });
    });
  }

  private async orderOpenTimeNow(
    hours: number,
    minutes: number,
    fcm: string,
    outletName: string,
    closeTime: Date,
    accountId: string,
  ) {
    const jobRule = new schedule.RecurrenceRule();
    const currentTime = new Date();

    currentTime.setHours(hours);
    currentTime.setMinutes(minutes);
    // for server 5.30 hrs
    currentTime.setMinutes(currentTime.getMinutes());
    // currentTime.setMinutes(currentTime.getMinutes() - 330);
    jobRule.hour = currentTime.getHours();
    jobRule.minute = currentTime.getMinutes();
    jobRule.tz = 'Asia/Kolkata'; // Set the time zone to Indian Standard Time (IST)

    schedule.scheduleJob(jobRule, () => {
      sendNotifications(
        fcm,
        'Order Open Now',
        'Hey ' +
          outletName +
          ' Your order is open till ' +
          closeTime +
          '. Place your order now.',
      );
      this.saveNotification({
        title: 'Order Open Now',
        desc:
          'Hey ' +
          outletName +
          ' Your order is open till ' +
          closeTime +
          '. Place your order now.',

        type: NotificationType.ORDER,
        redirectId: null,
        accountId,
      });
    });
  }

  private async orderCloseTime60After(
    hours: number,
    minutes: number,
    fcm: string,
    outletName: string,
    closeTime: Date,
    accountId: string,
  ) {
    const jobRule = new schedule.RecurrenceRule();
    const currentTime = new Date();

    currentTime.setHours(hours);
    currentTime.setMinutes(minutes);
    // for 60 min after + server 5.30 hrs
    // currentTime.setMinutes(currentTime.getMinutes() - 270);
    currentTime.setMinutes(currentTime.getMinutes() - 60);
    jobRule.hour = currentTime.getHours();
    jobRule.minute = currentTime.getMinutes();
    jobRule.tz = 'Asia/Kolkata'; // Set the time zone to Indian Standard Time (IST)

    schedule.scheduleJob(jobRule, () => {
      sendNotifications(
        fcm,
        'Order Closing Time',
        'Hey ' +
          outletName +
          ' Your order is closing soon in  ' +
          closeTime +
          '. Place your order now.',
      );
      this.saveNotification({
        title: 'Order Closing Time',
        desc:
          'Hey ' +
          outletName +
          ' Your order is closing soon in  ' +
          closeTime +
          '. Place your order now.',

        type: NotificationType.ORDER,
        redirectId: null,
        accountId,
      });
    });
  }

  private async orderCloseTime15After(
    hours: number,
    minutes: number,
    fcm: string,
    outletName: string,
    closeTime: Date,
    accountId: string,
  ) {
    const jobRule = new schedule.RecurrenceRule();
    const currentTime = new Date();

    currentTime.setHours(hours);
    currentTime.setMinutes(minutes);
    // for 15 min after + server 5.30 hrs
    // currentTime.setMinutes(currentTime.getMinutes() - 315);
    currentTime.setMinutes(currentTime.getMinutes() - 15);
    jobRule.hour = currentTime.getHours();
    jobRule.minute = currentTime.getMinutes();
    jobRule.tz = 'Asia/Kolkata'; // Set the time zone to Indian Standard Time (IST)

    // Create a job using the defined rule
    schedule.scheduleJob(jobRule, () => {
      sendNotifications(
        fcm,
        'Order Closing Time',
        'Hey ' +
          outletName +
          ' Your order is closing soon in  ' +
          closeTime +
          '. Place your order now.',
      );
      this.saveNotification({
        title: 'Order Closing Time',
        desc:
          'Hey ' +
          outletName +
          ' Your order is closing soon in  ' +
          closeTime +
          '. Place your order now.',

        type: NotificationType.ORDER,
        redirectId: null,
        accountId,
      });
    });
  }

  private saveNotification(dto: CreateNotifyDto) {
    return this.repo.save(dto);
  }
}
