import { HttpService } from '@nestjs/axios';
import { Injectable, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { NotificationType } from 'src/enum';
import { Repository } from 'typeorm';
import { Notification } from './entities/notification.entity';
import { OutletDetail } from 'src/outlet-details/entities/outlet-detail.entity';
import { PaginationReportDto } from './dto/notification.dto';

@Injectable()
export class NotificationsService {
  constructor(
    @InjectRepository(Notification)
    private readonly repo: Repository<Notification>,
    @InjectRepository(OutletDetail)
    private readonly repolet: Repository<OutletDetail>,
    private readonly httpService: HttpService,
  ) {}

  async sendBulkNotification(body, title, token, status) {
    const header = {
      headers: {
        'cache-control': 'no-cache',
        authorization: 'key=' + process.env.NOTIFY_AUTH,
        'content-type': 'application/json',
      },
    };
    let data = null;
    if (status) {
      data = JSON.stringify({
        registration_ids: token,
        data: { body: body, title: title, sound: 'default', type: 1 },
        notification: { body: body, title: title, sound: 'default', type: 1 },
      });
    } else {
      data = JSON.stringify({
        to: token,
        data: { body: body, title: title, sound: 'default', type: 1 },
        notification: { body: body, title: title, sound: 'default', type: 1 },
      });
    }

    try {
      const result = await this.httpService.axiosRef.post(
        `https://fcm.googleapis.com/fcm/send`,
        data,
        header,
      );
      if (result.data) {
        return result.data;
      }
    } catch (error) {
      return false;
    }
  }

  async create(createDto) {
    const result = await this.repo.count({
      where: {
        title: createDto.title,
        desc: createDto.desc,
        type: createDto.type,
        accountId: createDto.accountId,
      },
    });

    if (!result) {
      return this.repo.save(createDto);
    } else {
      return true;
    }
  }

  async findAll(limit: number, offset: number, accountId: string) {
    const [result, total] = await this.repo
      .createQueryBuilder('notification')
      .leftJoinAndSelect('notification.account', 'account')
      .where(
        'notification.accountId = :accountId',
        {
          accountId: accountId,
        },
      )
      .skip(offset)
      .take(limit)
      .orderBy({ 'notification.createdAt': 'DESC' })
      .getManyAndCount();
    return { result, total };
  }

  async findAllForAdmin(limit: number, offset: number) {
    const [payload, total] = await this.repo
      .createQueryBuilder('notification')
      .leftJoinAndSelect('notification.account', 'account')
      .where('notification.type IN (:...type)', {
        type: [
          NotificationType.CONTACT_US,
          NotificationType.TICKET,
          NotificationType.TIME_INCREMENT,
        ],
      })
      .skip(offset)
      .take(limit)
      .orderBy({ 'notification.createdAt': 'DESC' })
      .getManyAndCount();

      // push outlet name
      if(payload){
        let result = [];
        for (let index = 0; index < payload.length; index++) {
          const element = payload[index];
          const dataresult = await this.repolet.findOne({ where: { id: element.redirectId } });
          if(dataresult){
            result.push({
                id: element.id,
                title: element.title,
                desc: element.desc,
                type: element.type,
                redirectId: element.redirectId,
                read: element.read,
                createdAt: element.createdAt,
                accountId: element.accountId,
                account: element.account,
                storeName: dataresult.storeName,
              });
            } else {
            result.push({
                id: element.id,
                title: element.title,
                desc: element.desc,
                type: element.type,
                redirectId: element.redirectId,
                read: element.read,
                createdAt: element.createdAt,
                accountId: element.accountId,
                account: element.account,
                storeName: 'None',
              });
            }
          }
          
        return { result, total };
      } else {
        return { result:[], total };
      }
  }

  async update(id: number, accountId: string, status: boolean) {
    const notifications = await this.repo.findOne({ where: { id } });
    if (!notifications) {
      throw new NotFoundException('Notification not found!');
    }
    const obj = Object.assign(notifications, { read: status });
    return this.repo.save(obj);
  }

  async findReportAll(dto: PaginationReportDto) {
    const fromDate = new Date(dto.fromDate);
    fromDate.setHours(0, 0, 0, 0);
  
    const toDate = new Date(dto.toDate);
    toDate.setHours(23, 59, 59, 999);
  
    // total req
    const payloadtotal = await this.repo
      .createQueryBuilder('notification')
      .select([
        'notification.redirectId',
        'notification.accountId',
        'COUNT(notification.id) AS totalReq'
      ])
      .where('notification.type = :type and notification.createdAt >= :fromDate and notification.createdAt <= :toDate AND notification.accountId IS NULL', {
        type: NotificationType.TIME_INCREMENT,
        fromDate: fromDate,
        toDate: toDate
      })
      .groupBy('notification.redirectId')
      .take(dto.limit)
      .skip(dto.offset)
      .getRawMany();
  
    // total approved
    const payloadapproved = await this.repo
      .createQueryBuilder('notification')
      .select([
        'notification.redirectId',
        'notification.accountId',
        'COUNT(notification.id) AS totalApproved'
      ])
      .where('notification.type = :type and notification.createdAt >= :fromDate and notification.createdAt <= :toDate AND notification.accountId IS NOT NULL', {
        type: NotificationType.TIME_INCREMENT,
        fromDate: fromDate,
        toDate: toDate
      })
      .groupBy('notification.redirectId, notification.accountId')
      .take(dto.limit)
      .skip(dto.offset)
      .getRawMany();

    // push final list
    if(payloadtotal){
      let result = [];
      for (let index = 0; index < payloadtotal.length; index++) {
        const totalreq = payloadtotal[index];
        const totalapproved = payloadapproved[index];

        const dataresult = await this.repolet.findOne({ where: { id: totalreq.notification_redirectId } });
        result.push({
          storeName: dataresult.storeName,
          totalReqCount: totalreq.totalReq,
          totalApprovedCount: totalapproved.totalApproved,
          totalPendingCount: totalreq.totalReq - totalapproved.totalApproved,
        });
      }
      return { result, 'total':  result.length };
    } else {
      return { result: [], 'total': 0 };
    }
  }
}
