import {
  ConflictException,
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Repository } from 'typeorm';
import { CategoryDto } from './dto/category.dto';
import { Category } from './entities/category.entity';
import { BoolStatusDto } from 'src/common/dto/bool-status.dto';

@Injectable()
export class CategoryService {
  constructor(
    @InjectRepository(Category) private readonly repo: Repository<Category>,
  ) {}

  async create(dto: CategoryDto) {
    const category = await this.repo.findOne({ where: { name: dto.name } });
    if (category) {
      throw new ConflictException('Category already exists!');
    }
    const obj = Object.create(dto);
    return this.repo.save(obj);
  }

  async findAll(
    limit: number,
    offset: number,
    keyword: string,
    status: boolean,
  ) {
    const [result, total] = await this.repo
      .createQueryBuilder('category')
      .where('category.status = :status', { status: status })
      .andWhere(
        new Brackets((qb) => {
          qb.where('category.name LIKE :pname', {
            pname: '%' + keyword + '%',
          });
        }),
      )
      .orderBy(
        `CASE WHEN category.name LIKE '${keyword}%' THEN 0 ELSE 1 END, category.name`,
        'ASC',
      )
      .take(limit)
      .skip(offset)
      .getManyAndCount();

    return { result, total };
  }

  async find(limit: number, offset: number, keyword: string) {
    const [result, total] = await this.repo
      .createQueryBuilder('category')
      .where('category.status = :status', { status: true })
      .andWhere(
        new Brackets((qb) => {
          qb.where('category.name LIKE :pname', {
            pname: '%' + keyword + '%',
          });
        }),
      )
      .orderBy(
        `CASE WHEN category.name LIKE '${keyword}%' THEN 0 ELSE 1 END, category.name`,
        'ASC',
      )
      .take(limit)
      .skip(offset)
      .getManyAndCount();
    return { result, total };
  }

  async findOne(id: string) {
    const category = await this.repo.findOne({ where: { id } });
    if (!category) {
      throw new NotFoundException('Category not found!');
    }
    return category;
  }

  async update(id: string, dto: CategoryDto) {
    try {
      const category = await this.repo.findOne({ where: { id } });
      if (!category) {
        throw new NotFoundException('Category not found!');
      }
      const obj = Object.assign(category, { name: dto.name });
      return this.repo.save(obj);
    } catch (error) {
      throw new NotAcceptableException(
        'Either catgeory exists or invalid name!',
      );
    }
  }

  async image(image: string, result: Category) {
    const obj = Object.assign(result, {
      image: process.env.ST_CDN_LINK + 'category/' + image,
      imageName: image,
    });

    return this.repo.save(obj);
  }

  async status(id: string, dto: BoolStatusDto) {
    const menu = await this.repo.findOne({ where: { id } });
    if (!menu) {
      throw new NotFoundException('Category not found!');
    }
    const obj = Object.assign(menu, dto);
    return this.repo.save(obj);
  }
}
