"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.HostSink = void 0;
const rxjs_1 = require("rxjs");
const sink_1 = require("./sink");
class HostSink extends sink_1.SimpleSinkBase {
    constructor(_host, _force = false) {
        super();
        this._host = _host;
        this._force = _force;
        this._filesToDelete = new Set();
        this._filesToRename = new Set();
        this._filesToCreate = new Map();
        this._filesToUpdate = new Map();
    }
    _validateCreateAction(action) {
        return this._force ? rxjs_1.EMPTY : super._validateCreateAction(action);
    }
    _validateFileExists(p) {
        if (this._filesToCreate.has(p) || this._filesToUpdate.has(p)) {
            return (0, rxjs_1.of)(true);
        }
        if (this._filesToDelete.has(p)) {
            return (0, rxjs_1.of)(false);
        }
        for (const [from, to] of this._filesToRename.values()) {
            switch (p) {
                case from:
                    return (0, rxjs_1.of)(false);
                case to:
                    return (0, rxjs_1.of)(true);
            }
        }
        return this._host.exists(p);
    }
    _overwriteFile(path, content) {
        this._filesToUpdate.set(path, content);
        return rxjs_1.EMPTY;
    }
    _createFile(path, content) {
        this._filesToCreate.set(path, content);
        return rxjs_1.EMPTY;
    }
    _renameFile(from, to) {
        this._filesToRename.add([from, to]);
        return rxjs_1.EMPTY;
    }
    _deleteFile(path) {
        if (this._filesToCreate.has(path)) {
            this._filesToCreate.delete(path);
            this._filesToUpdate.delete(path);
        }
        else {
            this._filesToDelete.add(path);
        }
        return rxjs_1.EMPTY;
    }
    _done() {
        // Really commit everything to the actual filesystem.
        return (0, rxjs_1.concat)((0, rxjs_1.from)([...this._filesToDelete.values()]).pipe((0, rxjs_1.concatMap)((path) => this._host.delete(path))), (0, rxjs_1.from)([...this._filesToRename.entries()]).pipe((0, rxjs_1.concatMap)(([_, [path, to]]) => this._host.rename(path, to))), (0, rxjs_1.from)([...this._filesToCreate.entries()]).pipe((0, rxjs_1.concatMap)(([path, buffer]) => this._host.write(path, buffer))), (0, rxjs_1.from)([...this._filesToUpdate.entries()]).pipe((0, rxjs_1.concatMap)(([path, buffer]) => this._host.write(path, buffer)))).pipe((0, rxjs_1.reduce)(() => { }));
    }
}
exports.HostSink = HostSink;
//# sourceMappingURL=data:application/json;base64,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